% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/families.R
\name{custom_family}
\alias{custom_family}
\alias{customfamily}
\title{Custom Families in \pkg{brms} Models}
\usage{
custom_family(name, dpars = "mu", links = "identity", type = c("real",
  "int"), lb = NA, ub = NA, vars = NULL, env = parent.frame())
}
\arguments{
\item{name}{Name of the custom family.}

\item{dpars}{Names of the distributional parameters of
the family. One parameter must be named \code{"mu"} and
the main formula of the model will correspond to that
parameter.}

\item{links}{Names of the link functions of the 
distributional parameters.}

\item{type}{Indicates if the response distribution is
continuous (\code{"real"}) or discrete (\code{"int"}).}

\item{lb}{Vector of lower bounds of the distributional 
parameters. Defaults to \code{NA} that is no lower bound.}

\item{ub}{Vector of upper bounds of the distributional 
parameters. Defaults to \code{NA} that is no upper bound.}

\item{vars}{Names of variables, which are part of the likelihood
function without being distributional parameters. That is,
\code{vars} can be used to pass data to the likelihood. 
See \code{\link{stanvar}} for details about adding self-defined
data to the generated \pkg{Stan} model.}

\item{env}{An \code{\link{environment}} in which certain post-processing 
functions related to the custom family can be found. This is only
relevant if one wants to ensure compatibility with the methods
\code{\link[brms:predict.brmsfit]{predict}}, 
\code{\link[brms:fitted.brmsfit]{fitted}}, or
\code{\link[brms:log_lik.brmsfit]{log_lik}}.}
}
\value{
An object of class \code{customfamily} inheriting
  for class \code{\link{brmsfamily}}.
}
\description{
Define custom families for use in \pkg{brms} models.
It allows users to benefit from the modeling flexibility of 
\pkg{brms}, while applying their self-defined likelihood
functions. All of the post-processing methods for \code{brmsfit} 
objects can be made compatible with custom families. 
See \code{vignette("brms_customfamilies")} for more details.
For a list of built-in families see \code{\link{brmsfamily}}.
}
\details{
The corresponding probability density or mass \code{Stan} 
  functions need to have the same name as the custom family.
  That is if a family is called \code{myfamily}, then the 
  \pkg{Stan} functions should be called \code{myfamily_lpdf} or
  \code{myfamily_lpmf} depending on whether it defines a 
  continuous or discrete distribution.
}
\examples{
\dontrun{
## demonstrate how to fit a beta-binomial model
## generate some fake data
phi <- 0.7
n <- 300
z <- rnorm(n, sd = 0.2)
ntrials <- sample(1:10, n, replace = TRUE)
eta <- 1 + z
mu <- exp(eta) / (1 + exp(eta))
a <- mu * phi
b <- (1 - mu) * phi
p <- rbeta(n, a, b)
y <- rbinom(n, ntrials, p)
dat <- data.frame(y, z, ntrials)

# define a custom family
beta_binomial2 <- custom_family(
  "beta_binomial2", dpars = c("mu", "phi"),
  links = c("logit", "log"), lb = c(NA, 0),
  type = "int", vars = "trials[n]"
)

# define the corresponding Stan density function
stan_funs <- "
  real beta_binomial2_lpmf(int y, real mu, real phi, int N) {
    return beta_binomial_lpmf(y | N, mu * phi, (1 - mu) * phi);
  }
"

# fit the model
fit <- brm(y | trials(ntrials) ~ z, data = dat, 
           family = beta_binomial2, stan_funs = stan_funs)
summary(fit)
}

}
\seealso{
\code{\link{brmsfamily}}, \code{\link{stanvar}}
}
