#' Add variable labels
#'
#' Will add variable labels in a `var_label` column, based on:
#' 1. labels provided in `labels` argument if provided;
#' 2. variable labels defined in the original data frame with
#'    the `label` attribute (cf. [labelled::var_label()]);
#' 3. variable name otherwise.
#'
#' @details
#' If the `variable` column is not yet available in `x`,
#' [tidy_identify_variables()] will be automatically applied.
#'
#' It is possible to pass a custom label for an interaction
#' term in `labels` (see examples).
#' @param x a tidy tibble
#' @param labels an optional named list or named vector of
#' custom variable labels
#' @param interaction_sep separator for interaction terms
#' @param model the corresponding model, if not attached to `x`
#' @inheritParams tidy_plus_plus
#' @export
#' @family tidy_helpers
#' @examples
#' df <- Titanic %>%
#'   dplyr::as_tibble() %>%
#'   dplyr::mutate(Survived = factor(Survived, c("No", "Yes"))) %>%
#'   labelled::set_variable_labels(
#'     Class = "Passenger's class",
#'     Sex = "Sex"
#'   )
#'
#' df %>%
#'   glm(Survived ~ Class * Age * Sex, data = ., weights = .$n, family = binomial) %>%
#'   tidy_and_attach() %>%
#'   tidy_add_variable_labels(
#'     labels = list(Sex = "Gender", "Class:Age" = "Custom label")
#'   )
tidy_add_variable_labels <- function(x,
                                     labels = NULL,
                                     interaction_sep = " * ",
                                     model = tidy_get_model(x),
                                     quiet = FALSE,
                                     strict = FALSE) {
  if (is.null(model)) {
    stop("'model' is not provided. You need to pass it or to use 'tidy_and_attach()'.")
  }

  if ("header_row" %in% names(x)) {
    stop("`tidy_add_variable_labels()` cannot be applied after `tidy_add_header_rows().`")
  }

  if ("var_label" %in% names(x)) {
    x <- x %>% dplyr::select(-.data$var_label)
  }

  if (is.list(labels)) {
    labels <- unlist(labels)
  }

  if (!"variable" %in% names(x)) {
    x <- x %>% tidy_identify_variables(model = model)
  }

  # start with the list of variables
  variable_list <- model_list_variables(model)
  var_labels <- variable_list$variable
  names(var_labels) <- var_labels

  # identify terms with no variable
  # (e.g. intercepts)
  # use term name in that case

  variable_is_na <- is.na(x$variable)
  # temporarily copy term in variable
  x$variable[variable_is_na] <- x$term[variable_is_na]
  additional_labels <- unique(x$variable[variable_is_na])
  names(additional_labels) <- additional_labels
  var_labels <- var_labels %>%
    .update_vector(additional_labels)

  # table into account variable label attribute
  additional_labels <- variable_list$label_attr[!is.na(variable_list$label_attr)]
  names(additional_labels) <- variable_list$variable[!is.na(variable_list$label_attr)]
  var_labels <- var_labels %>%
    .update_vector(additional_labels)

  # check if all elements of labels are in x
  # show a message otherwise
  not_found <- setdiff(names(labels), names(var_labels))
  if (length(not_found) > 0 && !quiet) {
    usethis::ui_oops(paste0(
      usethis::ui_code(not_found),
      " terms have not been found in ",
      usethis::ui_code("x"),
      "."
    ))
  }
  if (length(not_found) > 0 && strict) {
    stop("Incorrect call with `labels=`. Quitting execution.", call. = FALSE)
  }

  var_labels <- var_labels %>%
    .update_vector(labels)

  # management of interaction terms
  interaction_terms <- x$variable[!is.na(x$var_type) & x$var_type == "interaction"]
  # do not treat those specified in labels
  interaction_terms <- setdiff(interaction_terms, names(labels))
  names(interaction_terms) <- interaction_terms
  # compute labels for interaction terms
  interaction_terms <- interaction_terms %>%
    strsplit(":") %>%
    lapply(function(x) {
      paste(var_labels[x], collapse = interaction_sep)
    }) %>%
    unlist()
  var_labels <- var_labels %>% .update_vector(interaction_terms)

  x %>%
    dplyr::left_join(
      tibble::tibble(
        variable = names(var_labels),
        var_label = var_labels
      ),
      by = "variable"
    ) %>%
    # restore missing values in variable
    dplyr::mutate(
      variable = dplyr::if_else(variable_is_na, NA_character_, .data$variable)
    ) %>%
    tidy_attach_model(model = model) %>%
    .order_tidy_columns()
}
