% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/models.R
\name{ssm_sde}
\alias{ssm_sde}
\title{Univariate state space model with continuous SDE dynamics}
\usage{
ssm_sde(
  y,
  drift,
  diffusion,
  ddiffusion,
  obs_pdf,
  prior_pdf,
  theta,
  x0,
  positive
)
}
\arguments{
\item{y}{Observations as univariate time series (or vector) of length
\eqn{n}.}

\item{drift, diffusion, ddiffusion}{An external pointers for the C++ functions
which
define the drift, diffusion and derivative of diffusion functions of SDE.}

\item{obs_pdf}{An external pointer for the C++ function which
computes the observational log-density given the the states and parameter
vector theta.}

\item{prior_pdf}{An external pointer for the C++ function which
computes the prior log-density given the parameter vector theta.}

\item{theta}{Parameter vector passed to all model functions.}

\item{x0}{Fixed initial value for SDE at time 0.}

\item{positive}{If \code{TRUE}, positivity constraint is
forced by \code{abs} in Milstein scheme.}
}
\value{
An object of class \code{ssm_sde}.
}
\description{
Constructs an object of class \code{ssm_sde} by defining the functions for
the drift, diffusion and derivative of diffusion terms of univariate SDE,
as well as the log-density of observation equation. We assume that the
observations are measured at integer times (missing values are allowed).
}
\details{
As in case of \code{ssm_nlg} models, these general models need a bit more
effort from the user, as you must provide the several small C++ snippets
which define the model structure. See vignettes for an example and
\code{cpp_example_model}.
}
\examples{

\donttest{ # Takes a while on CRAN
library("sde")
set.seed(1)
# theta_0 = rho = 0.5
# theta_1 = nu = 2
# theta_2 = sigma = 0.3
x <- sde.sim(t0 = 0, T = 50, X0 = 1, N = 50,
       drift = expression(0.5 * (2 - x)),
       sigma = expression(0.3),
       sigma.x = expression(0))
y <- rpois(50, exp(x[-1]))

# source c++ snippets
pntrs <- cpp_example_model("sde_poisson_OU")

sde_model <- ssm_sde(y, pntrs$drift, pntrs$diffusion,
 pntrs$ddiffusion, pntrs$obs_density, pntrs$prior,
 c(rho = 0.5, nu = 2, sigma = 0.3), 1, positive = FALSE)

est <- particle_smoother(sde_model, L = 12, particles = 500)

ts.plot(cbind(x, est$alphahat, 
  est$alphahat - 2*sqrt(c(est$Vt)), 
  est$alphahat + 2*sqrt(c(est$Vt))), 
  col = c(2, 1, 1, 1), lty = c(1, 1, 2, 2))


# Takes time with finer mesh, parallelization with IS-MCMC helps a lot
out <- run_mcmc(sde_model, L_c = 4, L_f = 8, 
  particles = 50, iter = 2e4,
  threads = 4L)
}

}
