% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bvs}
\alias{bvs}
\title{Bayesian Variable Selection}
\usage{
bvs(y, z, a, lambda, sigma_i, prob_prior, include = NULL)
}
\arguments{
\item{y}{a \eqn{K \times T} matrix of the endogenous variables.}

\item{z}{a \eqn{KT \times M} matrix of explanatory variables.}

\item{a}{an M-dimensional vector of parameter draws. If time varying parameters are used,
an \eqn{M \times T} coefficient matrix can be provided.}

\item{lambda}{an \eqn{M \times M} inclusion matrix that should be updated.}

\item{sigma_i}{the inverse variance-covariance matrix. If the variance-covariance matrix
is time varying, a \eqn{KT \times K} matrix can be provided.}

\item{prob_prior}{an M-dimensional vector of prior inclusion probabilities.}

\item{include}{an integer vector specifying the positions of variables, which should be
included in the BVS algorithm. If \code{NULL} (default), BVS will be applied to all variables.}
}
\value{
A matrix of inclusion parameters on its diagonal.
}
\description{
\code{bvs} employs Bayesian variable selection as proposed by Korobilis (2013)
to produce a vector of inclusion parameters for the coefficient matrix
of a VAR model.
}
\details{
The function employs Bayesian variable selection as proposed
by Korobilis (2013) to produce a vector of inclusion parameters, which are
the diagonal elements of the inclusion matrix \eqn{\Lambda} for the VAR model
\deqn{y_t = Z_t \Lambda a_t + u_t,}
where \eqn{u_t \sim N(0, \Sigma_{t})}.
\eqn{y_t} is a K-dimensional vector of endogenous variables and
\eqn{Z_t = x_t^{\prime} \otimes I_K} is a \eqn{K \times M} matrix of regressors with
\eqn{x_t} as a vector of regressors.
}
\examples{
# Prepare data
data("e1")
data <- diff(log(e1))
temp <- gen_var(data, p = 2, deterministic = "const")
y <- temp$Y
x <- temp$Z
z <- kronecker(t(x), diag(1, nrow(y)))
t <- ncol(y)
m <- nrow(y) * nrow(x)

# Priors
a_mu_prior <- matrix(0, m)
a_v_i_prior <- diag(0.1, m)

# Prior for inclusion parameter
prob_prior <- matrix(0.5, m)

# Initial value of Sigma
sigma <- tcrossprod(y) / t
sigma_i <- solve(sigma)

lambda <- diag(1, m)

z_bvs <- z \%*\% lambda

a <- post_normal_sur(y = y, z = z_bvs, sigma_i = sigma_i,
                     a_prior = a_mu_prior, v_i_prior = a_v_i_prior)

lambda <- bvs(y = y, z = z, a = a, lambda = lambda,
              sigma_i = sigma_i, prob_prior = prob_prior)

}
\references{
Korobilis, D. (2013). VAR forecasting using Bayesian variable selection. \emph{Journal of Applied Econometrics, 28}(2), 204--230. \url{https://doi.org/10.1002/jae.1271}
}
