#' Stacking of hourly temperatures
#' 
#' This function processes hourly temperatures generated by make_hourly_temps
#' for calculation of chilling and forcing. The chilling function requires
#' temperatures to be in a long list, and this function prepares them in this
#' way.
#' 
#' 
#' @param weather weather data frame containing either daily minimum ("Tmin")
#' and maximum ("Tmax") temperatures in the format generated by fix_weather, of
#' hourly temperatures in the format generated by make_hourly_temps (see below;
#' this can also be passed as hour_file).
#' @param latitude the geographic latitude (in decimal degrees) of the location
#' of interest
#' @param hour_file this is a data frame of hourly temperatures, as generated
#' by make_hourly_temps. It has columns describing the date (Year+JDay or
#' Year+Month+Day) and 24 columns called Hour_1 ... Hour_24 that contain hourly
#' temperatures. This is no longer required, since weather can be specified by
#' the weather argument. This parameter is only for compatibility with earlier
#' versions of chillR
#' @return list containing two elements: hourtemps: data frame containing all
#' the columns of the input data frame, except the hourly temperatures.
#' Instead, two columns are added: Hour is the hour of the day, and Temp is the
#' corresponding modeled mean temperature for that hour. QC: either the Quality
#' control attribute ("QC") passed into the function within the daily
#' temperature record produced by fix_weather, or NA.
#' @author Eike Luedeling
#' @references Luedeling E, Kunz A and Blanke M, 2013. Identification of
#' chilling and heat requirements of cherry trees - a statistical approach.
#' International Journal of Biometeorology 57,679-689.
#' @keywords utility
#' @examples
#' 
#' 
#' weather<-fix_weather(KA_weather[which(KA_weather$Year>2004),])
#' 
#' hourtemps<-stack_hourly_temps(weather, latitude=50.4)
#' 
#' 
#' 
#' @export stack_hourly_temps
stack_hourly_temps <-
function(weather=NULL,latitude=50,hour_file=NULL)
   {if(is.null(weather)&(!is.null(hour_file))) weather<-hour_file
     #this part is for the case where users specify and named variable 'hour_file'
     #This is necessary for backward compatibility
    if((length(names(weather))==2) & ("weather" %in% names(weather)) & ("QC" %in% names(weather))) 
      {THourly<-make_hourly_temps(latitude,weather$weather)
      QC<-weather$QC}
    if((length(names(weather))>24) & ("Hour_23" %in% names(weather)))
      {THourly<-weather
      QC<-NA}
   if((length(names(weather))>2) & ("Tmax" %in% names(weather)))
      {THourly<-make_hourly_temps(latitude,weather)
      QC<-NA}   
    
    if(is.null(weather)&is.null(hour_file)) {tt=NA;QC=NA} else
      {preserve_columns<-colnames(THourly)[1:(ncol(THourly)-24)]
      THourly[,"index"]<-row(THourly)[,1]
      ss<-stack(THourly,select=c(paste("Hour_",1:24,sep="")))
      colnames(ss)<-c("Temp","Hour")
      ss$Hour<-as.numeric(sapply(strsplit(as.character(ss$Hour),"_"),"[",2))
      for (pc in preserve_columns)
      {ss[,pc]<-rep(THourly[,pc],24)}
      tt<-ss[with(ss, order(Year, JDay,Hour)), ]  
      tt<-tt[c(preserve_columns,"Hour","Temp")]}
    
    return(list(hourtemps=tt,QC=QC))  
  }
          
    
