#' Download weather data from online database
#' 
#' This function retrieves either a list of nearby weather stations for a
#' specified point location, or it downloads weather data for a specific
#' weather station.
#' 
#' weather databases, from which chillR can download data: NOAA NCDC Global
#' Summary of the Day - "GSOD"
#' (https://data.noaa.gov/dataset/global-surface-summary-of-the-day-gsod)
#' 
#' California Irrigation Management Information System (CIMIS) - "CIMIS"
#' (http://www.cimis.water.ca.gov/)
#' 
#' University of California Integrated Pest Management (UCIPM) - "UCIPM"
#' (http://ipm.ucdavis.edu/WEATHER/)
#' 
#' several formats are possible for specifying the location vector, which can
#' consist of either two or three coordinates (it can include elevation).
#' Possible formats include c(1,2,3), c(1,2), c(x=1,y=2,z=3),
#' c(lat=2,long=1,elev=3). If elements of the vector are not names, they are
#' interpreted as c(Longitude, Latitude, Elevation).
#' 
#' The 'chillRCode' is generated by this function, when it is run with
#' geographic coordinates as location inputs. In the list of nearby stations
#' that is returned then, the chillRCode is provided and can then be used as
#' input for running the function in 'downloading' mode. For downloading the
#' data, use the same call as before but replace the location argument with the
#' chillRCode.
#' 
#' @param location either a vector of geographic coordinates, or the
#' 'chillRcode' of a weather station in the specified database. See details.
#' @param time_interval numeric vector with two elements, specifying the start
#' and end date of the period of interest.
#' @param database the database to be accessed. Must be "GSOD", "CIMIS"
#' or "UCIPM". Since among these, "UCIPM" is the most
#' comprehensive one for California, the initial area of interest, this is the
#' default.
#' @param station_list if the list of weather stations has already been
#' downloaded, the list can be passed to the function through this argument.
#' This can save a bit of time, since it can take a bit of time to download the
#' list, which can have several MB.
#' @param stations_to_choose_from if the location is specified by geographic
#' coordinates, this argument determines the number of nearby stations in the
#' list that is returned.
#' @param end_at_present boolean variable indicating whether the interval of
#' interest should end on the present day, rather than extending until the end
#' of the year specified under time_interval[2] (if time_interval[2] is the
#' current year).
#' @return The output depends on how the location is provided. If it is a
#' coordinate vector, the function returns a list of station_to_choose_from
#' weather stations that are close to the specified location. This list also
#' contains information about how far away these stations are (in km), how much
#' the elevation difference is (if elevation is specified; in m) and how much
#' overlap there is between the data contained in the database and the time
#' period specified by time_interval.
#' @note Many databases have data quality flags, which may sometimes indicate
#' that data aren't reliable. These are not considered by this function!
#' 
#' see the documentation of the handler functions (e.g. handle_ucipm) for details.
#' @author Eike Luedeling
#' @references The chillR package:
#' 
#' Luedeling E, Kunz A and Blanke M, 2013. Identification of chilling and heat
#' requirements of cherry trees - a statistical approach. International Journal
#' of Biometeorology 57,679-689.
#' @keywords utilities
#' @examples
#' 
#' 
#' #stat_list<-handle_gsod(action="list_stations",location=c(x=-122,y=38.5),
#' #  time_interval=c(2002,2002))
#' #the line above takes longer to run than CRAN allows for examples. The line below therefore
#' #generates an abbreviated stat_list that allows running the code.
#' stat_list<-data.frame(chillR_code=c("724828_99999","724828_93241","720576_174"),
#'    Lat=c(38.383,38.378,38.533),Long=c(-121.967,-121.958,-121.783),
#'    BEGIN=c(20010811,20060101,20130101),END=c(20051231,20160110,20160109))
#' 
#' #gw<-get_weather(location="724828_93241",time_interval=c(2012,2012),database="GSOD",
#' #   station_list = stat_list)
#' 
#' #stat_list<-get_weather(location=c(lat=50,lon=10,ele=150),time_interval=c(2001,2001),
#' #  database="UCIPM")
#' #chillRcode<-stat_list[which(stat_list$Perc_interval_covered==
#' #max(stat_list$Perc_interval_covered)),"chillR_code"][1]
#'   #after the first few lines here, the code should be "CEDARVIL.C"
#' #gw<-get_weather(location="CEDARVIL.C",time_interval=c(2001,2001),database="UCIPM")
#' #weather<-weather2chillR(gw,"GSOD")
#' #make_chill_plot(tempResponse(stack_hourly_temps(fix_weather(weather))),
#' #                "Chill_Portions",start_year=2005,end_year=2011,metriclabel="Chill Portions")
#' 
#' @export get_weather
get_weather<-function(location,time_interval=NA,database="UCIPM",station_list=NULL,stations_to_choose_from=25,end_at_present=TRUE)
{
  options(stringsAsFactors = FALSE)
  if(length(location)==2|length(location==3)) #determine whether location is specified by coordinates. If so move to station selection function
    if(is.numeric(location[1])&is.numeric(location[2]))
    {
      sorted_list<-switch(database,
                          GSOD = handle_gsod("list_stations",location=location,time_interval=time_interval,station_list=station_list,
                                             stations_to_choose_from=stations_to_choose_from,end_at_present=end_at_present),
                          CIMIS = handle_cimis("list_stations",location=location,time_interval=time_interval,station_list=station_list,
                                               stations_to_choose_from=stations_to_choose_from,end_at_present=end_at_present),
                          UCIPM = handle_ucipm("list_stations",location=location,time_interval=time_interval,station_list=station_list,
                                               stations_to_choose_from=stations_to_choose_from,end_at_present=end_at_present))
      if(is.null(sorted_list)) warning("No valid database specified") else
           sorted_list<-sorted_list[1:stations_to_choose_from,]
        
      return(sorted_list)
   }
  
  #if location is a string, we'll test if it belongs to a station in the database and, if so, try to retrieve weather data
  if(length(location)==1&is.character(location))
  {
    weather<-switch(database,
                        GSOD = handle_gsod("download_weather",location=location,time_interval=time_interval,station_list=station_list,
                                           end_at_present=end_at_present),
                        CIMIS = handle_cimis("download_weather",location=location,time_interval=time_interval,station_list=station_list,
                                             end_at_present=end_at_present),
                        UCIPM = handle_ucipm("download_weather",location=location,time_interval=time_interval,station_list=station_list,
                                             end_at_present=end_at_present))
    if(is.null(weather)) warning("No valid database specified")
    
    return(weather)
    }
  
}
           
 
