% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cfData-plotMethods.R
\name{plot,cfWind,missing-method}
\alias{plot,cfWind,missing-method}
\alias{plot.cfWind}
\alias{direction_plot,cfWind,missing-method}
\alias{direction_plot}
\alias{direction_plot,cfDataList,missing-method}
\alias{direction_plot,cfDataList,numeric-method}
\alias{speed_plot,cfWind,missing-method}
\alias{speed_plot}
\alias{speed_plot,cfDataList,missing-method}
\alias{speed_plot,cfDataList,numeric-method}
\title{Plot Clifro Wind Objects}
\usage{
\S4method{plot}{cfWind,missing}(
  x,
  y,
  n_directions = 12,
  n_speeds = 5,
  speed_cuts = NULL,
  col_pal = "GnBu",
  ggtheme = c("grey", "gray", "bw", "linedraw", "light", "minimal", "classic"),
  n_col = 1,
  ...
)

\S4method{direction_plot}{cfWind,missing}(
  x,
  y,
  ggtheme = c("grey", "gray", "bw", "linedraw", "light", "minimal", "classic"),
  contours = 10,
  n_col = 1,
  ...
)

\S4method{direction_plot}{cfDataList,missing}(x, y, ...)

\S4method{direction_plot}{cfDataList,numeric}(x, y, ...)

\S4method{speed_plot}{cfWind,missing}(
  x,
  y,
  ggtheme = c("grey", "gray", "bw", "linedraw", "light", "minimal", "classic"),
  scales = c("fixed", "free_x", "free_y", "free"),
  n_col = 1,
  ...
)

\S4method{speed_plot}{cfDataList,missing}(x, y, ...)

\S4method{speed_plot}{cfDataList,numeric}(x, y, ...)
}
\arguments{
\item{x}{a \code{cfWind} or \code{cfDataList} object.}

\item{y}{missing if \code{x} is a .\code{cfWind} object, otherwise a number
indicating the dataframe to plot in the \code{cfDataList} (defaults
to 1).}

\item{n_directions}{the number of direction bins to plot (petals on the
rose). The number of directions defaults to 12.}

\item{n_speeds}{the number of equally spaced wind speed bins to plot. This is
used if \code{spd_cuts} is NA (default 5).}

\item{speed_cuts}{numeric vector containing the cut points for the wind speed
intervals, or NA (default).}

\item{col_pal}{character string indicating the name of the
\code{RColorBrewer} colour palette to be
used for plotting, see 'Theme Selection' below.}

\item{ggtheme}{character string (partially) matching the
\code{\link[ggplot2]{ggtheme}} to be used for plotting, see
'Theme Selection' below.}

\item{n_col}{the number of columns of plots (default 1).}

\item{...}{further arguments passed to \code{\link[ggplot2]{theme}}.}

\item{contours}{the number of contour lines to draw (default 10).}

\item{scales}{character string partially matching the \code{scales} argument
in the \code{link[ggplot2]{facet_wrap}} function.}
}
\description{
Various plot methods for exploring wind speed and direction patterns for
given CliFlo stations.
}
\details{
If \code{x} is a \code{cfDataList}, by default the first datatype will be
plotted unless \code{y} is supplied.
}
\note{
If \code{x} is a \code{cfDataList} object and \code{y} refers to a
\pkg{clifro} dataframe that is not a \code{cfWind} object then it will be
passed to another method, if available.

The default \code{plot} method plots a different windrose for each CliFlo
station. The \code{direction_plot} method plots wind direction contours
through time to visualise temporal patterns in wind directions. The
\code{speed_plot} method plots the time series of wind speeds with a +/-
standard deviation region (if applicable).

Given a value on the x-axis, the ends of the density function along the
 y-axis are not constrained to be equal for any of the derivatives for the
 \code{direction_plot} method. That is, the contours at direction = 0, do not
 match the contours at direction = 360.

 @seealso \code{\link{plot,cfDataList,missing-method}} for general
  information on default plotting of \code{cfData} and \code{cfDataList}
  objects, and the links within. See \code{\link{cf_query}} for creating
  \code{cfWind} objects or \code{\link{windrose}} for plotting any wind data.
  Refer to \code{\link[ggplot2]{theme}} for more possible arguments to pass
  to these methods. \code{\link{summary,cfWind-method}} for summarising wind
  information at each CliFlo station.
}
\section{Theme Selection}{

For black and white windroses that may be preferred if plots are to be used
in journal articles for example, recommended \code{ggtheme}s are \code{'bw'},
\code{'linedraw'}, \code{'minimal'} or \code{'classic'} and
the \code{col_pal} should be \code{'Greys'}. Otherwise, any of the sequential
\code{RColorBrewer} colour palettes are recommended for
colour plots.
}

\examples{
\dontrun{
# Retrieve maximum wind gust data at the Reefton Ews station from CliFlo
# (public data)
reefton_wind = cf_query(cf_user(), cf_datatype(2, 2, 1, 1), cf_station(),
                        start_date = "2012-01-01-00")

class(reefton_wind)

# Examples of the default plots --------------------------------------------

# Plot a windrose
plot(reefton_wind)

# Plot the wind direction contours
direction_plot(reefton_wind)

# Plot the wind speed time-series
speed_plot(reefton_wind)

# Examples of changing the defaults ----------------------------------------

# Plot black and white windroses
plot(reefton_wind, ggtheme = "bw", col_pal = "Greys")
plot(reefton_wind, ggtheme = "linedraw", col_pal = "Greys")
plot(reefton_wind, ggtheme = "classic", col_pal = "Greys")
plot(reefton_wind, ggtheme = "minimal", col_pal = "Greys")

# Plot the wind directions using 20 contours and the ggtheme 'classic'
direction_plot(reefton_wind, ggtheme = "classic", contours = 20)

# Enlarge all the text to 18pt
library(ggplot2) # for element_text() and geom_point()
direction_plot(reefton_wind, ggtheme = "classic", contours = 20,
               text = element_text(size = 18))

# Include the actual observations in the plots
direction_plot(reefton_wind) + geom_point(alpha = .2, size = 3)

speed_plot(reefton_wind, ggtheme = "classic", text = element_text(size = 16)) +
  geom_point(shape = 1, size = 3)
# or equivalently using base graphics:
plot(reefton_wind$Date, reefton_wind$Speed, type = 'o',
     xlab = NA, ylab = "Daily max gust (m/s)", las = 1, main = "Reefton Ews")

# Example of plotting a cfDataList -----------------------------------------
# Collect both surface wind run and hourly surface wind observations from
# Reefton Ews
reefton_list = cf_query(cf_user(), cf_datatype(2, 1, 1:2, 1),
                        cf_station(), "2012-01-01 00", "2012-02-01 00")

reefton_list

class(reefton_list) #cfDataList

# Plot the first (default) dataframe
plot(reefton_list) # Error - no wind directions for wind run datatypes
# Try speed_plot instead
speed_plot(reefton_list)

# Plot the second dataframe in the cfDataList
plot(reefton_list, 2)           # identical to plot(reefton_list[2])
speed_plot(reefton_list, 2)     # identical to speed_plot(reefton_list[2])
direction_plot(reefton_list, 2) # identical to direction_plot(reefton_list[2])

# Save the ggplot externally -----------------------------------------------

# Save the plot as a png to the current working directory
library(ggplot2) # for ggsave()
ggsave("my_wind_plot.png")
}
}
