#'Test for auto-correlation in climate.
#'
#'Tests the correlation between the climate in a specified climate window and 
#'other fitted climate windows.
#'@param reference Reference climate data to be compared. Generated by functions
#'  \code{\link{singlewin}} or \code{\link{climatewin}}.
#'@param Xvar The climate variable of interest. Please specify the parent 
#'  environment and variable name (e.g. Climate$Temp).
#'@param CDate The climate date variable (dd/mm/yyyy). Please specify the parent
#'  environment and variable name (e.g. Climate$Date).
#'@param BDate The biological date variable (dd/mm/yyyy). Please specify the 
#'  parent environment and variable name (e.g. Biol$Date).
#'@param furthest The furthest number of time intervals (set by CINTERVAL) back 
#'  from the cutoff date or biological record to include in your climate window 
#'  search.
#'@param closest The closest number of time intervals (set by CINTERVAL) back 
#'  from the cutoff date or biological record to include in your climate window 
#'  search.
#'@param STAT The aggregate statistic used to analyse the climate data. Can 
#'  currently use basic R statistics (e.g. mean, min), as well as slope. 
#'  Additional aggregate statistics can be created using the format function(x) 
#'  (...). See parameter FUN in \code{\link{apply}} for more detail.
#'@param FIXED TRUE or FALSE, whether you wish the climate window to be variable
#'  (e.g. the number of days before each biological record is measured) or fixed
#'  (e.g. number of days before a set point in time).
#'@param cutoff.day,cutoff.month If FIXED is TRUE, the day and month of the 
#'  year from which the fixed window analysis will start.
#'@param CMISSING TRUE or FALSE, determines what should be done if there are 
#'  missing climate data. If FALSE, the function will not run if missing climate
#'  data is encountered. If TRUE, any records affected by missing climate data 
#'  will be removed from climate window analysis.
#'@param CINTERVAL The resolution at which climate window analysis will be 
#'  conducted. May be days ("D"), weeks ("W"), or months ("M"). Note the units 
#'  of parameters 'furthest' and 'closest' will differ with the choice of 
#'  CINTERVAL.
#'@return Will return a data frame showing the correlation between the climate 
#'  in each fitted window and the chosen reference window.
#'@author Liam D. Bailey and Martijn van de Pol
#' @examples
#' \dontrun{
#' 
#' # Test for auto-correlation using 'Mass' and 'MassClimate' data frames
#' 
#' data(Mass)
#' data(MassClimate)
#' 
#' # Fit a single climate window using the datasets Mass and MassClimate.
#' 
#' single <- singlewin(Xvar = MassClimate$Temp, CDate = MassClimate$Date, BDate = Mass$Date,
#'                     baseline = lm(Mass$Mass ~ 1), furthest = 72, closest = 15,
#'                     STAT = "mean", FUNC = "L",
#'                     FIXED = TRUE, cutoff.day = 20, cutoff.month = 5,
#'                     CMISSING = FALSE, CINTERVAL = "D")            
#' 
#' # Test the autocorrelation between the climate in this single window and other climate windows.
#' 
#' auto <- autowin(reference = single[[2]]$temporary,
#'                 Xvar  = MassClimate$Temp, CDate = MassClimate$Date, BDate = Mass$Date,
#'                 furthest = 365, closest = 0, STAT = "mean",
#'                 FIXED = TRUE, cutoff.day = 20, cutoff.month = 5,
#'                 CMISSING = FALSE, CINTERVAL = "D")
#'                 
#' # View the output
#' head(auto)                 
#'}
#'        
#'@export


# LAST EDITED: 20/02/2015
# EDITED BY: LIAM
# NOTES: Tidy up coding

autowin <- function(reference, Xvar, CDate, BDate, furthest, 
                    closest,  STAT, FIXED, cutoff.day, cutoff.month, 
                    CMISSING = FALSE, CINTERVAL = "D"){
  
  print("Initialising, please wait...")
  pb         <- SetProgressBar(furthest, closest, STAT) 
  cont       <- DateConverter(BDate = BDate, CDate = CDate, Xvar = Xvar, 
                              CINTERVAL = CINTERVAL, FIXED = FIXED, 
                              cutoff.day = cutoff.day, cutoff.month = cutoff.month)
  MODNO      <- 1
  MODLIST    <- list()
  duration   <- (furthest - closest) + 1
  CMatrix    <- matrix(ncol = (duration), nrow = length(BDate))
  temporary1 <- matrix(ncol = 1, nrow = length(BDate), 1)

  # Create a matrix with the climate data from closest to furthest days
  # back from each biological record
  for (i in 1:length(BDate)){
    for (j in closest:furthest){
      k <- j - closest + 1
      CMatrix[i, k] <- cont$Xvar[match(cont$BIntNo[i] - j, cont$CIntNo)]    
    }
  }
  
  if (CMISSING == FALSE && length(which(is.na(CMatrix))) > 0){
    stop(c("Climate data Xvar file should not contain NA values: ", length(which(is.na(CMatrix))), 
           " NA value(s) found. Please add missing climate data or set CMISSING=TRUE"))
  }
  
  if (CMISSING == TRUE && length(which(is.na(CMatrix))) > 0){
    reference <- reference[complete.cases(CMatrix)]
    CMatrix   <- CMatrix[complete.cases(CMatrix), ]
  } 
  
  for (m in closest:furthest){
    for (n in 1:duration){
      if ( (m - n) >= (closest - 1)){  # do not use windows that overshoot the closest possible day in window   
        if (STAT != "slope" || n > 1){
          windowopen  <- m - closest + 1
          windowclose <- windowopen-n + 1
          if (STAT == "slope"){ 
            time       <- seq(1, n, 1)
            temporary1 <- apply(CMatrix[, windowclose:windowopen], 1, FUN = function(x) coef(lm(x ~ time))[2])
          } else { 
            if (n == 1){
              temporary1 <- CMatrix[, windowclose:windowopen]
            } else {
              temporary1 <- apply(CMatrix[, windowclose:windowopen], 1, FUN = STAT)
            }
          }
          modeloutput <- cor(temporary1, reference)
          
          MODLIST$cor[[MODNO]]         <- modeloutput
          MODLIST$WindowOpen[[MODNO]]  <- m
          MODLIST$WindowClose[[MODNO]] <- m - n + 1
          MODNO                        <- MODNO + 1
        }
      }
    }  
    #Fill progress bar
    setTxtProgressBar(pb, MODNO - 1)
  }
  
  MODLIST$furthest    <- furthest
  MODLIST$closest     <- closest
  MODLIST$Statistics  <- STAT
  MODLIST$FIXED       <- FIXED
  
  if (FIXED == TRUE){
    MODLIST$cutoff.day   <- cutoff.day
    MODLIST$cutoff.month <- cutoff.month 
  }
  
  LocalAutoCor <- as.data.frame(MODLIST)
  return(LocalAutoCor)
}