#' @title Balance Statistics for `WeightIt` Objects
#' 
#' @description
#' Generates balance statistics for `weightit` and `weightitMSM` objects from \pkg{WeightIt}.
#' 
#' @inheritParams bal.tab
#' @param x a `weightit` or `weightitMSM` object; the output of a call to \pkgfun{WeightIt}{weightit} or \pkgfun{WeightIt}{weightitMSM}.
#' @param distance an optional formula or data frame containing distance values (e.g., propensity scores) or a character vector containing their names. If a formula or variable names are specified, `bal.tab()` will look in the argument to `data`, if specified. Propensity scores generated by `weightit()` and `weightitMSM()` are automatically included and named "prop.score".
#' @param s.d.denom `character`; how the denominator for standardized mean differences should be calculated, if requested. See [col_w_smd()] for allowable options. Abbreviations allowed. If not specified, `bal.tab()` will figure out which one is best based on the estimand of the `weightit` object: if ATT, `"treated"`; if ATC, `"control"`; otherwise `"pooled"`.
#' @param s.weights Optional; either a vector containing sampling weights for each unit or a string containing the name of the sampling weight variable in `data`. These function like regular weights except that both the adjusted and unadjusted samples will be weighted according to these weights if weights are used. If `s.weights` was supplied in the call to `weightit()` or `weightitMSM()`, they will automatically be included and do not need be specified again (though there is no harm if they are).
#' 
#' @returns
#' For point treatments, if clusters and imputations are not specified, an object of class `"bal.tab"` containing balance summaries for the `weightit` object. See [bal.tab()] for details.
#' 
#' If imputations are specified, an object of class `"bal.tab.imp"` containing balance summaries for each imputation and a summary of balance across imputations. See [`class-bal.tab.imp`] for details.
#' 
#' If `weightit()` is used with multi-category treatments, an object of class `"bal.tab.multi"` containing balance summaries for each pairwise treatment comparison. See [`bal.tab.multi()`][class-bal.tab.multi] for details.
#' 
#' If `weightitMSM()` is used for longitudinal treatments, an object of class `"bal.tab.msm"` containing balance summaries for each time period. See [`class-bal.tab.msm`] for details.
#' 
#' If clusters are specified, an object of class `"bal.tab.cluster"` containing balance summaries within each cluster and a summary of balance across clusters. See [`class-bal.tab.cluster`] for details.
#' 
#' @details
#' `bal.tab.weightit()` generates a list of balance summaries for the `weightit` object given.
#' 
#' @seealso
#' * [bal.tab()] for details of calculations.
#' 
#' @examplesIf all(sapply(c("WeightIt"), requireNamespace, quietly = TRUE))
#' library(WeightIt)
#' data("lalonde", package = "cobalt")
#' 
#' ## Basic propensity score weighting
#' w.out1 <- weightit(treat ~ age + educ + race + 
#'                        married + nodegree + re74 + re75, 
#'                    data = lalonde, method = "glm")
#' bal.tab(w.out1, un = TRUE, m.threshold = .1, 
#'         v.threshold = 2)
#' 
#' ## Weighting with a multi-category treatment
#' w.out2 <- weightit(race ~ age + educ + married + 
#'                        nodegree + re74 + re75, 
#'                    data = lalonde, method = "glm",
#'                    estimand = "ATE")
#' bal.tab(w.out2, un = TRUE)
#' bal.tab(w.out2, un = TRUE, pairwise = FALSE)
#' 
#' ## IPW for longitudinal treatments
#' data("msmdata", package = "WeightIt")
#' 
#' wmsm.out <- weightitMSM(list(A_1 ~ X1_0 + X2_0,
#'                         A_2 ~ X1_1 + X2_1 +
#'                             A_1 + X1_0 + X2_0,
#'                         A_3 ~ X1_2 + X2_2 +
#'                             A_2 + X1_1 + X2_1 +
#'                             A_1 + X1_0 + X2_0),
#'                    data = msmdata,
#'                    method = "glm")
#' bal.tab(wmsm.out)

#' @exportS3Method bal.tab weightit
bal.tab.weightit <- function(x,
                             stats, int = FALSE, poly = 1, distance = NULL, addl = NULL, data = NULL, continuous, binary, s.d.denom, thresholds = NULL, weights = NULL, cluster = NULL, imp = NULL, pairwise = TRUE, s.weights = NULL, abs = FALSE, subset = NULL, quick = TRUE,
                             ...) {
    
    tryCatch(args <- c(as.list(environment()), list(...))[-1], error = function(e) .err(conditionMessage(e)))
    
    #Adjustments to arguments
    
    args[vapply(args, rlang::is_missing, logical(1L))] <- NULL
    args[vapply(args, is_null, logical(1L)) & names(args) %nin% names(match.call())[-1]] <- NULL
    
    #Initializing variables
    X <- do.call("x2base", c(list(x), args), quote = TRUE)
    
    args[names(args) %in% names(X)] <- NULL
    
    X <- .assign_X_class(X)
    
    do.call("base.bal.tab", c(list(X), args),
            quote = TRUE)
}
