\encoding{utf8}
\name{copBasic.fitpara}
\alias{copBasic.fitpara.beta}
\title{ A Single or Multi-Parameter Optimization Engine (Beta Version) }
\description{
An example of a general implementation of a parameter optimization scheme using core features of the \pkg{copBasic} package. Because of the general complexity of the objectives for this function, the interface shown here is considered an \dQuote{beta version} and nomenclature is subject to dramatic changes in the future.
}
\usage{
copBasic.fitpara.beta(uv=NULL, popstat=NULL, statf=NULL, cop=NULL,
                      paradim=1, interval=NULL, par.init=NULL, ...)
}
\arguments{
  \item{uv}{An \R two column \code{matrix} or \code{data.frame} of a sample of nonexceedance probabilities \eqn{u} and \eqn{v};}
  \item{popstat}{The population statistic(s) that will be used if \code{uv} is \code{NULL};}
  \item{statf}{A function responsible at the minimum for computation of the theoretical values of the population statistic(s) that the optimization will revolve around; This function, if supporting an \code{as.sample} interface (\emph{e.g.} \code{\link{hoefCOP}}) and if \code{uv} has been provided, will be dispatched to compute the population statistic(s);}
  \item{cop}{A copula function that is passed along to \code{statf} though of course the \code{statf} function can decide whether to use this argument or not;}
  \item{paradim}{The dimension of the parameters. In reality, the default triggers uni-dimensional root solving using the \code{uniroot()} function in \R or otherwise the \code{optim()} function in \R is used for multi-dimensional minimization with subtle changes in setup (see source code). Alternative logic could be have been used but it is felt that this is the most logical for future adaptations;}
  \item{interval}{The \code{interval} argument by the same name for the \code{uniroot()} function;}
  \item{par.init}{The initial parameter vector for the \code{optim()} function; and}
  \item{...}{Additional arguments to pass.}
}
\value{
  A vector of the values for the parameter variable is returned
}
\note{
\emph{One-Parameter Optimization}---A demonstration of one-dimensional parameter optimimization using \emph{Gini's Gamma} (\code{\link{giniCOP}}), which is a measure of bivariate association. There is no native support for Gini's Gamma (and most of the other measures of association) in regards to being a parameter estimator at the copula function interface level in \pkg{copBasic}. (A converse example is one provided internally by the \code{\link{GHcop}} copula.)
\preformatted{
  set.seed(24); n <- 230 # sample size to draw from Galambos copula but a
  # different copula was chosen for the fitting.
  sampleUV <- simCOP(n=n, cop=GLcop, para=1.5) # a random sample
  para <- copBasic.fitpara.beta(uv=sampleUV, statf=giniCOP,
                                interval=c(.1,200), cop=HRcop) # 1.959521
}

\emph{Three-Parameter Optimization}---A demonstration of multi-dimensional parameter optimimization using Gini Gamma (\code{\link{giniCOP}}), \emph{Joe Nu-Skew} (\code{\link{nuskewCOP}}), and \emph{Joe Nu-Skew-Star} (\code{\link{nustarCOP}}). This is substantially more complicated to implement. The \emph{\enc{Hüsler}{Husler}--Reiss copula} (\code{\link{HRcop}}) is chosen both as part of the sample simulation for the study as well as the copula as part of the modeling.  Using composition by the \code{\link{composite1COP}}, first establish a parent three parameter copula and simulate from it to create a bivariate sample in \code{sampleUV} that will be used for demonstration. A standard normal variate graphic of the simulation is generated by \code{\link{simCOP}} as well---later, additional results will be superimposed.
\preformatted{
  n <- 610; set.seed(1999) # Seed value will be used again (see below)
  pop.para <- list(cop1=HRcop, para1=4, alpha=0.14, beta=0.35)
  sampleUV <- simCOP(n=n, cop=composite1COP, para=pop.para, col=3, snv=TRUE)
}

A custom objective function \code{objfunc} to serve as the \code{statf} for the \code{copBasic.fitpara.beta} call. The objective function has the \code{as.sample} interface (\emph{e.g.} \code{\link{giniCOP}}) implemented for sample estimation. The most subtle feature of function presumably is the use of the \code{pnorm()} function in \R for the \eqn{\alpha} and \eqn{\beta} parameters to keep each parameter in the range \eqn{\alpha, \beta \in (0,1)}. Another subtly, which affects the choice of other copulas from \code{\link{HRcop}}, is how the parameter range of \eqn{\Theta} (the \code{para[1]} variable) is controlled---here the parameter remains untransformed but the lower domain edge is truncated by the return of infinity (\code{return(Inf)}). The \code{getstat} argument is only for short circuiting the objective so that \code{objfunc} can be used to compute the three statistics after the optimal parameters are found.
\preformatted{
  "objfunc" <- function(para, stat=NA, as.sample=FALSE, getstat=FALSE, ...) {
      if(as.sample) {
         return(c(giniCOP(  para=para, as.sample=TRUE),
                  nuskewCOP(para=para, as.sample=TRUE),
                  nustarCOP(para=para, as.sample=TRUE)))
      }
      para[1]   <- ifelse(para[1] < 0, return(Inf), para[1]) # edge check
      para[2:3] <-  pnorm(para[2:3]) # detransform
      para <- list(cop1=HRcop, para1=para[1], alpha=para[2], beta=para[3])
      hp <- c(giniCOP(  composite1COP, para),
              nuskewCOP(composite1COP, para),
              nustarCOP(composite1COP, para))
      if(getstat) return(hp) # short circuit to get the statistics out
      dv <- stat; dv[dv == 0] <- 1 # changing dv "adapts" the error to
      return(sqrt(sum(((stat-hp)/dv)^2))) # trap division by zero
   }
}

The parameter estimation proceeds in the following code. The sample statistics (or \code{target.stats}) are computed and subsequently passed to the optimizaiton using the \code{popstat} argument. Note the use of the \code{qnorm()} function in \R to transform the initial guess \eqn{\alpha = \beta = 1/2} into a domain more easily handled by the optimizer (\code{optim()} function in \R). The transformed optimal parameters are computed, and then the values of the three statistics for the fit are computed. Lastly, a \pkg{copBasic} parameter object \code{fit.para} is created, which can be used for later comparisons.
\preformatted{
  txt <- c("GiniGamma", "JoeNuSkew", "JoeNuSkewStar")
  target.stats <- objfunc(sampleUV,as.sample=TRUE); names(target.stats) <-txt
  raw.fit.para <- copBasic.fitpara.beta(popstat=target.stats, statf=objfunc,
         par.init=c(1,qnorm(.5),qnorm(.5)), cop=composite1COP, paradim=3)
  fit.stats <- objfunc(raw.fit.para, getstat=TRUE); names(fit.stats) <-txt
  fit.para <- list(cop1=HRcop, para1=raw.fit.para[1],
                   alpha=pnorm(raw.fit.para[2]), beta=pnorm(raw.fit.para[3]))
}

The optimization is completed. It can be informative to see what the simulation of the fitted copula looks like. Note that this particular example suffers from identifiability problems between the parameters---meaning that local minima exist or that satisfactory solutions using different parameters than used to generate the random sample can occur. The same seed is used so that one-to-one comparison of points can visually be made with the \code{\link{simCOP}} function call.
\preformatted{
  set.seed(1999) # This value will be used again (see below)
  sampleUV <- simCOP(n=n, cop=composite1COP, para=fit.para,
                ploton=FALSE, pch=16, cex=0.5, col=2, snv=TRUE) # red dots
}

The visual comparison is qualitative. The tabular comparison of the sample statistics to those of the fitted model shows that perhaps an acceptable local minima has been attained in terms of \dQuote{fit} but the subsequent comparison of the parameters of the population used to generate the sample and those of the fitted model seemingly depart substantially in the \eqn{\alpha \rightarrow 0} parameter of the copula composition. The tail dependency parameters are similar, but further goodness-of-fit check is not made.
\preformatted{
                       #    GiniGamma     JoeNuSkew  JoeNuSkewStar
  print(target.stats)  #   0.52190271   -0.01214717     0.01219315
  print(fit.stats)     #   0.49442298   -0.01198534     0.01061656

                          # Parameter  Alpha       Beta
  print(ls.str(pop.para)) #      4.00   0.14      0.350  # given
  print(ls.str(fit.para)) #      2.48   2.25e-08  0.355  # one solution

                                               # Tail Dependency Parameters
  taildepCOP(cop=composite1COP, para=pop.para) # lower = 0 : upper = 0.5838
  taildepCOP(cop=composite1COP, para=fit.para) # lower = 0 : upper = 0.5302
}

The demonstration ends with the comparison of the two asymmetrical density contours.
\preformatted{
  densityCOPplot(cop=composite1COP, para=pop.para, contour.col=3)
  densityCOPplot(cop=composite1COP, para=fit.para, contour.col=2,
                                    ploton=FALSE)
}
}
\author{ W.H. Asquith}
\examples{
# See the Note section
}
\keyword{copula (fitting)}
\keyword{copula (goodness-of-fit)}
\keyword{copula (inference)}
\keyword{inference}
\keyword{goodness-of-fit}
