% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/corr2d.R
\name{corr2d}
\alias{corr2d}
\title{Two-dimensional correlation analysis.}
\usage{
corr2d(Mat1, Mat2 = NULL, Ref1 = NULL, Ref2 = NULL, Wave1 = NULL,
  Wave2 = NULL, Time = NULL, Int = stats::splinefun,
  N = 2^ceiling(log2(NROW(Mat1))), Norm = 1/(pi * (NCOL(Mat1) - 1)),
  scaling = 0, corenumber = parallel::detectCores(), preview = FALSE)
}
\arguments{
\item{Mat1, Mat2}{Numeric matrix containing the data which will be correlated;
'\emph{Spectral variable}' by columns and '\emph{perturbation}' by rows. For hetero
correlations \code{Mat1} and \code{Mat2} must have the same number of rows.}

\item{Ref1, Ref2}{Numeric vector containg a single spectrum, which will be
substracted from \code{Mat1} (or \code{Mat2}, respectivly) to generate dynamic spectra
for 2D correlation analysis. Defaults to \code{NULL} in which case the \code{colMeans()}
of \code{Mat1} (or \code{Mat2}, respectivly) is used as reference. The length of \code{Ref1}
(or \code{Ref2}) needs to be equal to the number of columns in \code{Mat1} (or \code{Mat2}).}

\item{Wave1, Wave2}{Numeric vector containing the spectral variable. Needs to be
specified if column names of \code{Mat1} (or \code{Mat2}) are undefinied.}

\item{Time}{Numeric vector containing the perturbation. If specified, \code{Mat1}
(and \code{Mat2} if given) will be interpolated to \code{N} equally spaced perturbation
values using \code{Int} to speed up the fft algorithm.}

\item{Int}{Function specifing how the dataset will be interpolated to give
\code{N} equally spaced perturbation values.}

\item{N}{Positive, non-zero integer specifing how many equally spaced
perturbation values should be interpolated using \code{Int}. \code{corr2d}
is fastest if \code{N} is a power of 2.}

\item{Norm}{Numeric vector specifing how the correlation matrix should be
normalized. Needs to have length \code{(NCOL(Mat1) - 1)}.}

\item{scaling}{Positive real number used as exponent when scaling the dataset
with its standard deviation. Defaults to 0 meaning no scaling. 0.5
(\emph{Pareto scaling}) and 1 (\emph{Pearson scaling}) are commonly used to enhance
weak correlations relative to strong correlations.}

\item{corenumber}{Positive, non-zero integer specifing how many CPU cores
should be used for parallel fft computation.}

\item{preview}{Logical scalar, should a 3D preview of the synchronous
correlation spectrum be drawn at the end? Uses \code{\link[rgl]{persp3d}} from \pkg{rgl}
package.}
}
\value{
\code{corr2D} returns a list of class "corr2d" containing the complex
    correlation matrix (\code{$FT}), the used reference spectra (\code{$Ref1},
    \code{$Ref2}), the spectral variables (\code{$Wave1}, \code{$Wave2}), the
    (interpolated) perturbation (\code{$Time}) and logical scalar (\code{$Het})
    indicating if homo (\code{FALSE}) or hetero (\code{TRUE}) correlation was done.
}
\description{
\code{corr2d} calculates the synchronous and asynchronous correlation
    spectra between \code{Mat1} and \code{Mat1} (homo correlation)
    or between \code{Mat1} and \code{Mat2} (hetero correlation).
}
\details{
\code{corr2d} uses a parallel fast Fourier transformation
    (\code{\link[stats]{fft}}) to calculate the complex correlation matrix.
    For parallelization the \code{\link[foreach]{foreach}} function is used.
    Large input matrices (> 4000 columns) can lead to long calculation times
    depending on the number of cores used. Also note that the resulting
    matrix can become very large, adjust the RAM limit with
    \code{\link[utils]{memory.limit}} accordingly. For a detailed description
    of the underlying math see references.
}
\examples{
    data(FuranMale, package = "corr2D")
    twod <- corr2d(FuranMale, Ref1 = FuranMale[1, ], corenumber = 1)
    
    plot_corr2d(twod, xlab = expression(paste("relative Wavenumber" / cm^-1)),
                      ylab = expression(paste("relative Wavenumber" / cm^-1)))

}
\references{
I. Noda, \emph{Appl. Spectrosc.}, 1993, \strong{47}, 1329-1336.\cr
    I. Noda, \emph{Vib. Spectrosc.}, 2012, \strong{60}, 146-153.
}
\seealso{
For plotting of the resulting list containing the 2D correlation
    spectra see \code{\link{plot_corr2d}} and \code{\link{plot_corr2din3d}}.
}

