% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxmeg_m.R
\name{coxmeg_m}
\alias{coxmeg_m}
\title{Fit a Cox mixed-effects model for estimating HRs for a set of predictors}
\usage{
coxmeg_m(X, outcome, corr, FID = NULL, cov = NULL, tau = NULL,
  min_tau = 1e-04, max_tau = 5, eps = 1e-06, order = 1,
  detap = TRUE, opt = "bobyqa", eigen = TRUE, score = FALSE,
  dense = FALSE, threshold = 0, solver = 1, spd = TRUE,
  verbose = TRUE, mc = 100, invchol = TRUE)
}
\arguments{
\item{X}{A matrix of the preidctors. Can be quantitative or binary values. Categorical variables need to be converted to dummy variables. Each row is a sample, and the predictors are columns.}

\item{outcome}{A matrix contains time (first column) and status (second column). The status is a binary variable (1 for failure / 0 for censored).}

\item{corr}{A relatedness matrix. Can be a matrix or a 'dgCMatrix' class in the Matrix package. Must be symmetric positive definite or symmetric positive semidefinite.}

\item{FID}{An optional string vector of family ID. If provided, the data will be reordered according to the family ID.}

\item{cov}{An optional matrix of the covariates included in the null model for estimating the variance component. Can be quantitative or binary values. Categorical variables need to be converted to dummy variables. Each row is a sample, and the covariates are columns.}

\item{tau}{An optional positive value for the variance component. If \code{tau} is given, the function will skip estimating the variance component, and use the given \code{tau} to analyze the predictors.}

\item{min_tau}{An optional positive value indicating the lower bound in the optimization algorithm for the variance component \code{tau}. Default is 1e-4.}

\item{max_tau}{An optional positive value indicating the upper bound in the optimization algorithm for the variance component \code{tau}. Default is 5.}

\item{eps}{An optional positive value indicating the tolerance in the optimization algorithm. Default is 1e-6.}

\item{order}{An optional integer value starting from 0. Only valid when \code{dense=FALSE}. It specifies the order of approximation used in the inexact newton method. Default is 1.}

\item{detap}{An optional logical value indicating whether to use approximation for log-determinant. Default is TRUE.}

\item{opt}{An optional logical value for the Optimization algorithm for tau. Can have the following values: 'bobyqa', 'Brent' or 'NM'. Default is 'bobyqa'.}

\item{eigen}{An optional logical value. Only effective when \code{dense=FALSE}. It indicates whether to use RcppEigen:LDLT to solve linear systems. Default is TRUE.}

\item{score}{An optional logical value indicating whether to perform a score test. Default is FALSE.}

\item{dense}{An optional logical value indicating whether the relatedness matrix is dense. Default is FALSE.}

\item{threshold}{An optional non-negative value. If threshold>0, coxmeg_m will reestimate HRs for those SNPs with a p-value<threshold by first estimating a variant-specific variance component. Default is 0.}

\item{solver}{An optional bianry value taking either 1 or 2. Default is 1. See details.}

\item{spd}{An optional logical value indicating whether the relatedness matrix is symmetric positive definite. Default is TRUE. See details.}

\item{verbose}{An optional logical value indicating whether to print additional messages. Default is TRUE.}

\item{mc}{An optional integer value specifying the number of Monte Carlo samples used for approximating the log-determinant. Only valid when dense=TRUE and detap=TRUE. Default is 100.}

\item{invchol}{An optional logical value. Only effective when \code{dense=FALSE}. If TRUE, sparse Cholesky decomposition is used to compute the inverse of the relatedness matrix. Otherwise, sparse LU is used.}
}
\value{
beta: The estimated coefficient for each predictor in X.

HR: The estimated HR for each predictor in X.

sd_beta: The estimated standard error of beta.

p: The p-value of each SNP.

tau: The estimated variance component.

iter: The number of iterations until convergence.
}
\description{
\code{coxmeg_m} first estimates the variance component under a null model with only cov, and then analyzing each predictor in X one by one.
}
\section{About \code{spd}}{

When \code{spd=TRUE}, the relatedness matrix is treated as SPD. If the matrix is SPSD or not sure, use \code{spd=FALSE}.
}

\section{About \code{solver}}{

When solver=1/solver=2, Cholesky decompositon/PCG is used to solve the linear system. However, when \code{dense=FALSE} and \code{eigen=FALSE}, the solve function in the Matrix package is used regardless of \code{solver}. When \code{dense=TRUE}, it is recommended to set \code{solver=2} to have better computational performance.
}

\section{About \code{invchol}}{

Cholesky decomposition using \code{invchol=TRUE} is generally (but not always) much faster to invert a relatedness matrix (e.g., a block-diagonal matrix). But for some types of sparse matrices (e.g., a banded AR(1) matrix with rho=0.9), it sometimes can be very slow. In such cases, \code{invchol=FALSE} is can be used.
}

\examples{
library(Matrix)
library(MASS)
library(coxmeg)

## simulate a block-diagonal relatedness matrix
tau_var <- 0.2
n_f <- 100
mat_list <- list()
size <- rep(10,n_f)
offd <- 0.5
for(i in 1:n_f)
{
  mat_list[[i]] <- matrix(offd,size[i],size[i])
  diag(mat_list[[i]]) <- 1
}
sigma <- as.matrix(bdiag(mat_list))
n <- nrow(sigma)

## simulate random effects and outcomes
x <- mvrnorm(1, rep(0,n), tau_var*sigma)
myrates <- exp(x-1)
y <- rexp(n, rate = myrates)
cen <- rexp(n, rate = 0.02 )
ycen <- pmin(y, cen)
outcome <- cbind(ycen,as.numeric(y <= cen))

## simulate genotypes
g = matrix(rbinom(n*5,2,0.5),n,5)

## The following command will first estimate the variance component without g, 
## and then use it to estimate the HR for each preditor in g.
re = coxmeg_m(g,outcome,sigma,tau=0.5,order=1,eigen=TRUE,dense=FALSE)
re
}
\keyword{Cox}
\keyword{mixed-effects}
\keyword{model}
