% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxmeg_plink.R
\name{coxmeg_plink}
\alias{coxmeg_plink}
\title{Perform GWAS using a Cox mixed-effects model with plink files as input}
\usage{
coxmeg_plink(
  pheno,
  corr,
  type,
  bed = NULL,
  tmp_dir = NULL,
  cov_file = NULL,
  tau = NULL,
  maf = 0.05,
  min_tau = 1e-04,
  max_tau = 5,
  eps = 1e-06,
  order = NULL,
  detap = NULL,
  opt = "bobyqa",
  score = FALSE,
  threshold = 0,
  solver = NULL,
  spd = TRUE,
  mc = 100,
  verbose = TRUE
)
}
\arguments{
\item{pheno}{A string value indicating the file name or the full path of a pheno file. The files must be in the working directory if the full path is not given. The file is in plink pheno format, containing the following four columns, family ID, individual ID, time and status. The status is a binary variable (1 for events/0 for censored).}

\item{corr}{A relatedness matrix or a List object of matrices if there are multiple relatedness matrices. They can be a matrix or a 'dgCMatrix' class in the Matrix package. The matrix (or the sum if there are multiple) must be symmetric positive definite or symmetric positive semidefinite. The order of subjects must be consistent with that in pheno.}

\item{type}{A string indicating the sparsity structure of the relatedness matrix. Should be 'bd' (block diagonal), 'sparse', or 'dense'. See details.}

\item{bed}{A optional string value indicating the file name or the full path of a plink bed file (without .bed). The files must be in the working directory if the full path is not given. If not provided, only the variance component will be returned.}

\item{tmp_dir}{A optional directory to store temporary .gds files. The directory needs to be specified when \code{bed} is provided.}

\item{cov_file}{An optional string value indicating the file name or the full path of a covariate file. The files must be in the working directory if the full path is not given. Same as the cov file in plink, the first two columns are family ID and individual ID. The covariates are included in the null model for estimating the variance component. The covariates can be quantitative or binary values. Categorical variables need to be converted to dummy variables.}

\item{tau}{An optional positive value or vector for the variance component(s). If tau is given, the function will skip estimating the variance component, and use the given tau to analyze the SNPs.}

\item{maf}{An optional positive value. All SNPs with MAF<maf in the bed file will not be analyzed. Default is 0.05.}

\item{min_tau}{An optional positive value indicating the lower bound in the optimization algorithm for the variance component tau. Default is 1e-4.}

\item{max_tau}{An optional positive value indicating the upper bound in the optimization algorithm for the variance component tau. Default is 5.}

\item{eps}{An optional positive value indicating the relative convergence tolerance in the optimization algorithm. Default is 1e-6. A smaller value (e.g., 1e-8) can be used for better precision of the p-values in the situation where most SNPs under investigation have a very low minor allele count (<5).}

\item{order}{An optional integer value starting from 0. Only effective when \code{dense=FALSE}. It specifies the order of approximation used in the inexact newton method. Default is NULL, which lets coxmeg choose an optimal order.}

\item{detap}{An optional string indicating whether to use an approximation for log-determinant. Can be 'exact', 'diagonal', 'gkb', or 'slq'. Default is NULL, which lets the function select a method based on 'type' and other information. See details.}

\item{opt}{An optional logical scalar for the Optimization algorithm for estimating the variance component(s). Can be one of the following values: 'bobyqa', 'Brent', 'NM', or 'L-BFGS-B' (only for >1 variance components). Default is 'bobyqa'.}

\item{score}{An optional logical value indicating whether to perform a score test. Default is FALSE.}

\item{threshold}{An optional non-negative value. If threshold>0, coxmeg_m will reestimate HRs for those SNPs with a p-value<threshold by first estimating a variant-specific variance component. Default is 0.}

\item{solver}{An optional binary value taking either 1 or 2. Default is 1. See details.}

\item{spd}{An optional logical value indicating whether the relatedness matrix is symmetric positive definite. Default is TRUE. See details.}

\item{mc}{An optional integer scalar specifying the number of Monte Carlo samples used for approximating the log-determinant when \code{detap='gkb'} or \code{detap='slq'}. Default is 100.}

\item{verbose}{An optional logical value indicating whether to print additional messages. Default is TRUE.}
}
\value{
beta: The estimated coefficient for each predictor in X.

HR: The estimated HR for each predictor in X.

sd_beta: The estimated standard error of beta.

p: The p-value of each SNP.

tau: The estimated variance component.

rank: The rank of the relatedness matrix.

nsam: Actual sample size.
}
\description{
\code{coxmeg_plink} first estimates the variance component under a null model with only cov if tau is not given, and then analyzing each SNP in the plink files.
}
\section{About \code{corr}}{

The subjects in \code{corr} must be in the same order as in the plink fam file.
}

\section{About missing values}{

\code{pheno} -9 for missing values, \code{cov_file} NA for missing values.
}

\section{About temporary files}{

The function will create a temporary gds file with approximately the same size as the bed file. The temporary file will be removed when the analysis is done.
}

\section{About \code{type}}{

Specifying the type of the relatedness matrix (whether it is block-diagonal, general sparse, or dense). In the case of multiple relatedness matrices, it refers to the type of the sum of these matrices.
\itemize{ 
\item{"bd"}{ - used for a block-diagonal relatedness matrix, or a sparse matrix the inverse of which is also sparse. }
\item{"sparse"}{ - used for a general sparse relatedness matrix the inverse of which is not sparse.}
\item{"dense"}{ - used for a dense relatedness matrix.}
}
}

\section{About \code{spd}}{

When \code{spd=TRUE}, the relatedness matrix is treated as SPD. If the matrix is SPSD or not sure, use \code{spd=FALSE}.
}

\section{About \code{solver}}{

Specifying which method is used to solve the linear system in the optimization algorithm.  
\itemize{ 
\item{"1"}{ - Cholesky decompositon (RcppEigen:LDLT) is used to solve the linear system.}
\item{"2"}{ - PCG is used to solve the linear system. When \code{type='dense'}, it is recommended to set \code{solver=2} to have better computational performance.}
}
}

\section{About \code{detap}}{

Specifying the method to compute the log-determinant for estimating the variance component(s).
\itemize{ 
\item{"exact"}{ - the exact log-determinant is computed for estimating the variance component.}
\item{"diagonal"}{ - uses diagonal approximation and is only effective for a sparse relatedness matrix.}
\item{"slq"}{ - uses stochastic lanczos quadrature approximation. It uses the Lanczos algorithm to compute the weights and nodes. When type is 'bd' or 'sparse', it is often faster than 'gkb' and has the same accuracy. When type='dense', it is fater than 'gkb' by around half, but can be inaccurate if the relatedness matrix is (almost) singular.}
\item{"gkb"}{ - uses stochastic lanczos quadrature approximation. It uses the Golub-Kahan bidiagonalization algorithm to compute the weights and nodes. It is robust against an (almost) singular relatedness matrix when type='dense', but it is generally slower than 'slq'.}  
}
}

\examples{
library(Matrix)
library(MASS)
library(coxmeg)

## build a block-diagonal relatedness matrix
n_f <- 600
mat_list <- list()
size <- rep(5,n_f)
offd <- 0.5
for(i in 1:n_f)
{
  mat_list[[i]] <- matrix(offd,size[i],size[i])
  diag(mat_list[[i]]) <- 1
}
sigma <- as.matrix(bdiag(mat_list))

## Estimate variance component under a null model
pheno = system.file("extdata", "ex_pheno.txt", package = "coxmeg")
cov = system.file("extdata", "ex_cov.txt", package = "coxmeg")
bed = system.file("extdata", "example_null.bed", package = "coxmeg")
bed = substr(bed,1,nchar(bed)-4)
re = coxmeg_plink(pheno,sigma,type='bd',bed=bed,tmp_dir=tempdir(),cov_file=cov,
detap='diagonal',order=1)
re
}
\keyword{Cox}
\keyword{mixed-effects}
\keyword{model}
