#' Compute shortest distance between origin and destination nodes.
#' 
#' @param Graph  An object generated by cppRouting::makegraph() function.
#' @param from A vector of one or more vertices from which distances are calculated (origin).
#' @param to A vector of one or more vertices (destination).
#' @param algorithm character. "Dijkstra" for uni-directional Dijkstra, "bi" for bi-directional Dijkstra, "A*" or "NBA" for New bi-directional A star .Default to "Dijkstra"
#' @param constant numeric. Constant to maintain the heuristic function admissible in A* algorithm. 
#' Default to 1, when cost is expressed in the same unit than coordinates. See details
#' @param allcores Logical. If TRUE, all cores are used.
#' @return Vector of shortest distances.
#' @note 'from' and 'to' must be the same length.
#' @details To perform A*, projected coordinates should be provided in the Graph object.  
#' In A* algorithms, euclidean distance is used as heuristic function.
#' To understand how A star algorithm work, see https://en.wikipedia.org/wiki/A*_search_algorithm .
#' To understand the importance of constant parameter, see the package description : https://github.com/vlarmet/cppRouting
#' @examples 
#' #Data describing edges of the graph
#' edges<-data.frame(from_vertex=c(0,0,1,1,2,2,3,4,4), 
#'                   to_vertex=c(1,3,2,4,4,5,1,3,5), 
#'                   cost=c(9,2,11,3,5,12,4,1,6))
#' 
#' #Get all nodes
#' nodes<-unique(c(edges$from_vertex,edges$to_vertex))
#'  
#' #Construct directed and undirected graph 
#' directed_graph<-makegraph(edges,directed=TRUE)
#' non_directed<-makegraph(edges,directed=FALSE)
#' 
#' #Sampling origin and destination nodes
#' origin<-sample(nodes,10,replace=TRUE)
#' destination<-sample(nodes,10,replace=TRUE)
#' 
#' #Get distance between origin and destination in the two graphs
#' dir_dist<-get_distance_pair(Graph=directed_graph, from=origin, to=destination, allcores=FALSE)
#' non_dir_dist<-get_distance_pair(Graph=non_directed, from=origin, to=destination, allcores=FALSE)
#' print(dir_dist)
#' print(non_dir_dist)

get_distance_pair<-function(Graph,from,to,algorithm="Dijkstra",constant=1,allcores=FALSE){
  
  if (length(from)!=length(to)) stop("From and to have not the same length")
  
  if (any(is.na(cbind(from,to)))) stop("NAs are not allowed in origin/destination nodes")

  from<-as.character(from)
  from_id<-Graph$dict$id[match(from,Graph$dict$ref)]
  to<-as.character(to)
  to_id<-Graph$dict$id[match(to,Graph$dict$ref)]
  
  
  
  if (!is.null(Graph$coords)){
    if (algorithm %in% c("NBA","A*","bi")){
   if (algorithm=="A*"){
      message("Running A* ...")
      if (allcores==FALSE) res<-Astar(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,Graph$coords[,2],Graph$coords[,3],constant)
      else {
        numWorkers <- parallel::detectCores()
        cl <- parallel::makeCluster(numWorkers, type = "PSOCK")
        parallel::clusterEvalQ(cl = cl,library("cppRouting"))
        chunks <- parallel::splitIndices(length(from), ncl = numWorkers)
        mylist<-lapply(chunks,function(x) from_id[x])
        mylist2<-lapply(chunks,function(x) to_id[x])
        
        
        res<-parallel::clusterMap(cl,Astar,dep=mylist,arr=mylist2,
                        MoreArgs = list(gfrom=Graph$data$from,gto=Graph$data$to,gw=Graph$data$dist,NbNodes=Graph$nbnode,lat=Graph$coords$X,lon=Graph$coords$Y,k=constant))
        parallel::stopCluster(cl)
        res<-c(unlist(res))
        return(res)
      }
   }
    if (algorithm=="NBA"){
      message("Running NBA* ...")
      if (allcores==FALSE) res<-NBA(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,Graph$coords[,2],Graph$coords[,3],constant)
      else {
        numWorkers <- parallel::detectCores()
        cl <- parallel::makeCluster(numWorkers, type = "PSOCK")
        parallel::clusterEvalQ(cl = cl,library("cppRouting"))
        chunks <- parallel::splitIndices(length(from), ncl = numWorkers)
        mylist<-lapply(chunks,function(x) from_id[x])
        mylist2<-lapply(chunks,function(x) to_id[x])
        
        
        res<-parallel::clusterMap(cl,NBA,dep=mylist,arr=mylist2,
                                  MoreArgs = list(gfrom=Graph$data$from,gto=Graph$data$to,gw=Graph$data$dist,NbNodes=Graph$nbnode,lat=Graph$coords$X,lon=Graph$coords$Y,k=constant))
        parallel::stopCluster(cl)
        res<-c(unlist(res))
        return(res)
      }
    }
    
    
    if (algorithm=="bi"){
      message("Running bidirectional Dijkstra...")
      if (allcores==FALSE) res<-Bidir(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
      else {
        numWorkers <- parallel::detectCores()
        cl <- parallel::makeCluster(numWorkers, type = "PSOCK")
        parallel::clusterEvalQ(cl = cl,library("cppRouting"))
        chunks <- parallel::splitIndices(length(from), ncl = numWorkers)
        mylist<-lapply(chunks,function(x) from_id[x])
        mylist2<-lapply(chunks,function(x) to_id[x])
        res<-parallel::clusterMap(cl,Bidir,dep=mylist,arr=mylist2,
                        MoreArgs = list(gfrom=Graph$data$from,gto=Graph$data$to,gw=Graph$data$dist,NbNodes=Graph$nbnode))
        parallel::stopCluster(cl)
        res<-c(unlist(res))
        return(res)
      }
    }
    }
    
    else {
      message("Running Dijkstra ...")
      if (allcores==FALSE) res<-Dijkstra_early_stop(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
      else {
        numWorkers <- parallel::detectCores()
        cl <- parallel::makeCluster(numWorkers, type = "PSOCK")
        parallel::clusterEvalQ(cl = cl,library("cppRouting"))
        chunks <- parallel::splitIndices(length(from), ncl = numWorkers)
        mylist<-lapply(chunks,function(x) from_id[x])
        mylist2<-lapply(chunks,function(x) to_id[x])
        res<-parallel::clusterMap(cl,Dijkstra_early_stop,dep=mylist,arr=mylist2,
                                  MoreArgs = list(gfrom=Graph$data$from,gto=Graph$data$to,gw=Graph$data$dist,NbNodes=Graph$nbnode))
        parallel::stopCluster(cl)
        res<-c(unlist(res))
        return(res)
        }
    }
    
  }
  else {
    if (algorithm=="bi"){
      message("Running bidirectional Dijkstra...")
      if (allcores==FALSE) res<-Bidir(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
      else {
        numWorkers <- parallel::detectCores()
        cl <- parallel::makeCluster(numWorkers, type = "PSOCK")
        parallel::clusterEvalQ(cl = cl,library("cppRouting"))
        chunks <- parallel::splitIndices(length(from), ncl = numWorkers)
        mylist<-lapply(chunks,function(x) from_id[x])
        mylist2<-lapply(chunks,function(x) to_id[x])
        res<-parallel::clusterMap(cl,Bidir,dep=mylist,arr=mylist2,
                                  MoreArgs = list(gfrom=Graph$data$from,gto=Graph$data$to,gw=Graph$data$dist,NbNodes=Graph$nbnode))
        parallel::stopCluster(cl)
        res<-c(unlist(res))
        return(res)
      }
    }
    else {
    message("Running Dijkstra ...")
    if (allcores==FALSE) res<-Dijkstra_early_stop(from_id,to_id,Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode)
    else {
      numWorkers <- parallel::detectCores()
      cl <- parallel::makeCluster(numWorkers, type = "PSOCK")
      parallel::clusterEvalQ(cl = cl,library("cppRouting"))
      chunks <- parallel::splitIndices(length(from), ncl = numWorkers)
      mylist<-lapply(chunks,function(x) from_id[x])
      mylist2<-lapply(chunks,function(x) to_id[x])
      res<-parallel::clusterMap(cl,Dijkstra_early_stop,dep=mylist,arr=mylist2,
                                MoreArgs = list(gfrom=Graph$data$from,gto=Graph$data$to,gw=Graph$data$dist,NbNodes=Graph$nbnode))
      parallel::stopCluster(cl)
      res<-c(unlist(res))
      return(res)
    }
    }
  
  }  
  
  
  return(res)
  
  
}
