#' Reduce the number of edges by removing non-intersection nodes, duplicated edges and isolated loops in the graph.
#' 
#' @param Graph  An object generated by cppRouting::makegraph() function.
#' @param keep Character or integer vector. Nodes of interest that will not be removed. Default to NULL
#' @param new_edges Logical. If TRUE, new edges created by removing nodes are returned as character vectors of the following form : \emph{junction node - removed node(s) - junction node}. Default to FALSE
#' @param rm_loop Logical. if TRUE, isolated loops as removed. Default to TRUE
#' @param iterate Logical. If TRUE, process is repeated until only intersection nodes remain in the graph. Default to FALSE
#' @param silent Logical. If TRUE and iterate set to TRUE, number of iteration and number of removed nodes are printed to the console.
#' @return List of length two containing the simplified cppRouting graph and new edges
#' @details To understand why process can be iterated, see the package description : https://github.com/vlarmet/cppRouting  
#' 
#' The first iteration usually eliminates the majority of non-intersection nodes and is therefore faster.
#' Created edges at n-th iteration will be stored at n-th list of \emph{new_edges} list.
#' @examples 
#' #Simple directed graph
#' edges<-data.frame(from=c(1,2,3,4,5,6,7,8),
#'                   to=c(0,1,2,3,6,7,8,5),
#'                   dist=c(1,1,1,1,1,1,1,1))
#'                   
#' #Plot
#' if(requireNamespace("igraph",quietly = TRUE)){
#' igr<-igraph::graph_from_data_frame(edges)
#' plot(igr)
#' }
#' 
#' #Construct cppRouting graph
#' graph<-makegraph(edges,directed=TRUE)
#'                                     
#' #Simplify the graph, removing loop
#' simp<-cpp_simplify(graph, rm_loop=TRUE)
#' 
#' #Convert cppRouting graph to data frame
#' simp<-to_df(simp$graph)
#' 
#' #Plot
#' if(requireNamespace("igraph",quietly = TRUE)){
#' igr<-igraph::graph_from_data_frame(simp)
#' plot(igr)
#' }
#' 
#' #Simplify the graph, keeping node 2 and keeping loop
#' simp<-cpp_simplify(graph,keep=2 ,rm_loop=FALSE) 
#' 
#' #Convert cppRouting graph to data frame
#' simp<-to_df(simp$graph)
#' 
#' #Plot
#' if(requireNamespace("igraph",quietly = TRUE)){
#' igr<-igraph::graph_from_data_frame(simp)
#' plot(igr)
#' }
#' 

cpp_simplify<-function(Graph,keep=NULL,new_edges=FALSE,rm_loop=TRUE,iterate=FALSE,silent=TRUE){
  
  #Nodes to keep
  to_keep<-rep(0,Graph$nbnode)
  if (!is.null(keep)) {
    keep<-as.character(keep)
    to_keep[Graph$dict$ref %in% keep]<-1
  }
  
  
  simp<-Simplify2(Graph$data$from,Graph$data$to,Graph$data$dist,Graph$nbnode,loop=rm_loop,keep = to_keep,dict = Graph$dict$ref)
  
  
  
  if (new_edges==TRUE)edges<-list(simp[[3]])
  else edges<-NULL

  
  
  
  #Because removing nodes can create other nodes to remove
  counter<-1
  while(iterate==TRUE){
    if (counter==1 & silent==FALSE) message(paste("  iteration :",counter,"-",Graph$nbnode-simp[[2]],"nodes removed"))
    
    if (simp[[2]]==Graph$nbnode) break
    count<-simp[[2]]
    
    
    
    rd<-Remove_duplicate(simp[[1]][,1],simp[[1]][,2],simp[[1]][,3],Graph$nbnode)
    simp<-Simplify2(rd[,1],rd[,2],rd[,3],Graph$nbnode,loop=rm_loop,keep = to_keep,dict = Graph$dict$ref)
    
    counter<-counter+1
    
    
    
    
    if (count == simp[[2]]) break
    
    if(silent==FALSE) message(paste("  iteration :",counter,"-",count-simp[[2]],"nodes removed"))
    
    
    
    if (new_edges==TRUE) edges[[length(edges)+1]]<-simp[[3]]
    
  }
  
  rd<-Remove_duplicate(simp[[1]][,1],simp[[1]][,2],simp[[1]][,3],Graph$nbnode)

  
  simp<-rd
  if (nrow(simp)==0) stop("All nodes have been removed")
  
  Nodes=unique(c(simp[,1],simp[,2]))

  
  dict<-Graph$dict[Graph$dict$id %in% Nodes,]
  dict$idnew<-0:(nrow(dict)-1)
  simp[,1]<-dict$idnew[match(simp[,1],dict$id)]
  simp[,2]<-dict$idnew[match(simp[,2],dict$id)]
  simp<-as.data.frame(simp)
  simp[,1]<-as.integer(simp[,1])
  simp[,2]<-as.integer(simp[,2])
  colnames(simp)<-c("from","to","dist")
  if (!is.null(Graph$coords)){
    coords<-Graph$coords
    coords<-coords[match(dict$id,Graph$dict$id),]
  }
  else coords=NULL
  
  dict<-dict[,-2]
  colnames(dict)<-c("ref","id")
  


  
  return(list(graph=list(data=simp,
                         coords=coords,
                         nbnode=length(Nodes),
                         dict=dict),
              new_edges=edges))
  
  }
