% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cross_validate_fn.R
\name{cross_validate_fn}
\alias{cross_validate_fn}
\title{Cross-validate custom model functions for model selection}
\usage{
cross_validate_fn(data, model_fn, formulas, fold_cols = ".folds",
  type = "gaussian", cutoff = 0.5, positive = 2,
  predict_type = NULL, predict_fn = NULL, metrics = list(),
  rm_nc = FALSE, parallel = FALSE)
}
\arguments{
\item{data}{Data frame.

 Must include grouping factor for identifying folds
  - as made with \code{\link[groupdata2:fold]{groupdata2::fold()}}.}

\item{model_fn}{Model function that returns a fitted model object.
 Will usually wrap an existing model function like \code{\link[e1071:svm]{e1071::svm}}
 or \code{\link[nnet:multinom]{nnet::multinom}}.

 Must have the following function arguments:

 \code{function(train_data, formula)}}

\item{formulas}{Model formulas as strings. (Character)

 Will be converted to \code{\link[stats:formula]{formula}} objects
 before being passed to \code{model_fn}.

 E.g. \code{c("y~x", "y~z")}.

 Can contain random effects.

 E.g. \code{c("y~x+(1|r)", "y~z+(1|r)")}.}

\item{fold_cols}{Name(s) of grouping factor(s) for identifying folds. (Character)

 Include names of multiple grouping factors for repeated cross-validation.}

\item{type}{Type of evaluation to perform:

 \code{"gaussian"} for regression (like linear regression).

 \code{"binomial"} for binary classification.

 \code{"multinomial"} for multiclass classification.}

\item{cutoff}{Threshold for predicted classes. (Numeric)

 N.B. \strong{Binomial models only}}

\item{positive}{Level from dependent variable to predict.
 Either as character or level index (\code{1} or \code{2} - alphabetically).

 E.g. if we have the levels \code{"cat"} and \code{"dog"} and we want \code{"dog"} to be the positive class,
 we can either provide \code{"dog"} or \code{2}, as alphabetically, \code{"dog"} comes after \code{"cat"}.

 Used when calculating confusion matrix metrics and creating ROC curves.

 N.B. Only affects evaluation metrics, not the model training or returned predictions.

 N.B. \strong{Binomial models only}.}

\item{predict_type}{The \code{type} argument for \code{\link[stats:predict]{predict()}}.

 When the defaults fail, provide it such that the \code{\link[stats:predict]{predict()}}
 output is as follows:

 \subsection{Binomial}{
 Vector or one-column matrix / data frame with probabilities (0-1).
 E.g.:

 \code{c(0.3, 0.5, 0.1, 0.5)}
 }

 \subsection{Gaussian}{
 Vector or one-column matrix / data frame with the predicted value.
 E.g.:

 \code{c(3.7, 0.9, 1.2, 7.3)}
 }

 \subsection{Multinomial}{
 Data frame with one column per class containing probabilities of the class.
 Column names should be identical to how the class names are written in the target column.
 E.g.:

 \tabular{rrr}{
  \strong{class_1} \tab \strong{class_2} \tab
  \strong{class_3} \cr
  0.269 \tab 0.528 \tab 0.203\cr
  0.368 \tab 0.322 \tab 0.310\cr
  0.375 \tab 0.371 \tab 0.254\cr
  ... \tab ... \tab ...}

 N.B. \code{predict_fn} and \code{predict_type} are mutually exclusive. Specify only one of them.
 }}

\item{predict_fn}{Function for predicting the targets in the test folds using the fitted model object.
 Will usually wrap \code{\link[stats:predict]{predict()}}, but doesn't have to.
 Must return predictions in the format described in \code{predict_type} above.

 Must have the following function arguments:

 \code{function(test_data, model, formula = NULL)}

 N.B. \code{predict_fn} and \code{predict_type} are mutually exclusive. Specify only one of them.}

\item{metrics}{List for enabling/disabling metrics.

  E.g. \code{list("RMSE" = FALSE)} would remove RMSE from the results,
  and \code{list("Accuracy" = TRUE)} would add the regular accuracy metric
  to the classification results.
  Default values (TRUE/FALSE) will be used for the remaining metrics available.

  Also accepts the string \code{"all"}.

  N.B. Currently, disabled metrics are still computed.}

\item{rm_nc}{Remove non-converged models from output. (Logical)}

\item{parallel}{Whether to cross-validate the list of models in parallel. (Logical)

 Remember to register a parallel backend first.
 E.g. with \code{doParallel::registerDoParallel}.}
}
\value{
Tbl (tibble) with results for each model.

 \subsection{Shared across families}{
 A nested tibble with \strong{coefficients} of the models from all iterations. The coefficients
 are extracted from the model object with \code{\link[broom:tidy]{broom::tidy()}} or
 \code{\link[stats:coef]{coef()}} (with some restrictions on the output).
 If these attempts fail, a default coefficients tibble filled with \code{NA}s is returned.

 Number of \emph{total} \strong{folds}.

 Number of \strong{fold columns}.

 Count of \strong{convergence warnings}, using a limited set of keywords (e.g. "convergence"). If a
 convergence warning does not contain one of these keywords, it will be counted with \strong{other warnings}.
 Consider discarding models that did not converge on all iterations.
 Note: you might still see results, but these should be taken with a grain of salt!

 Nested tibble with the \strong{warnings and messages} caught for each model.

 Specified \strong{family}.

 Name of \strong{dependent} variable.

 Names of \strong{fixed} effects.

 Names of \strong{random} effects, if any.
 }

 ----------------------------------------------------------------

 \subsection{Gaussian Results}{

 ----------------------------------------------------------------

 Average \strong{RMSE}, \strong{MAE}, \strong{r2m}, \strong{r2c}, \strong{AIC}, \strong{AICc},
 and \strong{BIC} of all the iterations*,
 \emph{\strong{omitting potential NAs} from non-converged iterations}. Some metrics will
 return \code{NA} if they can't be extracted from the fitted model objects.

 N.B. The Information Criteria metrics (AIC, AICc, and BIC) are also averages.

 A nested tibble with the \strong{predictions} and targets.

 A nested tibble with the non-averaged \strong{results} from all iterations.

 * In \emph{repeated cross-validation},
 the metrics are first averaged for each fold column (repetition) and then averaged again.

 }

 ----------------------------------------------------------------

 \subsection{Binomial Results}{

 ----------------------------------------------------------------

 Based on the \strong{collected} predictions from the test folds*,
 a confusion matrix and a ROC curve are created to get the following:

 ROC:

 \strong{AUC}, \strong{Lower CI}, and \strong{Upper CI}

 Confusion Matrix:

 \strong{Balanced Accuracy}, \strong{F1},
 \strong{Sensitivity}, \strong{Specificity},
 \strong{Positive Prediction Value},
 \strong{Negative Prediction Value},
 \strong{Kappa},
 \strong{Detection Rate},
 \strong{Detection Prevalence},
 \strong{Prevalence}, and
 \strong{MCC} (Matthews correlation coefficient).

 Other available metrics (disabled by default, see \code{metrics}):
 \strong{Accuracy}.

 Also includes:

 A nested tibble with the \strong{predictions}, predicted classes (depends on \code{cutoff}), and targets.
 Note, that the \strong{predictions are not necessarily of the specified \code{positive} class}, but of
 the model's positive class (second level of dependent variable, alphabetically).

 A nested tibble with the sensativities and specificities from the \strong{ROC} curves.

 A nested tibble with the \strong{confusion matrix}/matrices.
 The \code{Pos_} columns tells you whether a row is a
 True Positive (TP), True Negative (TN), False Positive (FP), or False Negative (FN),
 depending on which level is the "positive" class. I.e. the level you wish to predict.

 A nested tibble with the \strong{results} from all fold columns, when using \emph{repeated cross-validation}.

 * In \emph{repeated cross-validation}, an evaluation is made per fold column (repetition) and averaged.

 }

 ----------------------------------------------------------------

 \subsection{Multinomial Results}{

 ----------------------------------------------------------------

 For each class, a \emph{one-vs-all} binomial evaluation is performed. This creates
 a \strong{class level results} tibble containing the same metrics as the binomial results
 described above, along with the \strong{Support} metric, which is simply a
 count of the class in the target column. These metrics are used to calculate the macro metrics
 in the output tibble. The nested class level results tibble is also included in the output tibble,
 and would usually be reported along with the macro and overall metrics.

 The output tibble contains the macro and overall metrics.
 The metrics that share their name with the metrics in the nested
 class level results tibble are averages of those metrics
 (note: does not remove \code{NA}s before averaging).
 In addition to these, it also includes the \strong{Overall Accuracy} metric.

 Other available metrics (disabled by default, see \code{metrics}):
 \strong{Accuracy}, \strong{Weighted Balanced Accuracy}, \strong{Weighted Accuracy},
 \strong{Weighted F1}, \strong{Weighted Sensitivity}, \strong{Weighted Sensitivity},
 \strong{Weighted Specificity}, \strong{Weighted Pos Pred Value},
 \strong{Weighted Neg Pred Value}, \strong{Weighted AUC}, \strong{Weighted Lower CI},
 \strong{Weighted Upper CI}, \strong{Weighted Kappa}, \strong{Weighted MCC},
 \strong{Weighted Detection Rate}, \strong{Weighted Detection Prevalence}, and
 \strong{Weighted Prevalence}.

 Note that the "Weighted" metrics are weighted averages, weighted by the \code{Support}.

 Also includes:

 A nested tibble with the \strong{predictions}, predicted classes, and targets.

 A nested tibble with the multiclass \strong{Confusion Matrix}.

 \strong{Class Level Results}

 The nested class level results tibble also includes:

 A nested tibble with the sensativities and specificities from the \strong{ROC} curve.

 A nested tibble with the \strong{confusion matrix} from the one-vs-all evaluation.
 The \code{Pos_} columns tells you whether a row is a
 True Positive (TP), True Negative (TN), False Positive (FP), or False Negative (FN),
 depending on which level is the "positive" class. In our case, \code{1} is the current class
 and \code{0} represents all the other classes together.

 }
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

 Cross-validate your model function with one or multiple model formulas at once.
 Perform repeated cross-validation.
 Returns results in a tibble for easy comparison,
 reporting and further analysis.

 Compared to \code{\link[cvms:cross_validate]{cross_validate()}},
 this function allows you supply a custom model function
 and (if needed) a custom predict function.

 Supports regression and classification (binary and multiclass). See \code{type}.

 Note that some metrics may not be computable for all types
 of model objects.
}
\details{
Packages used:

 \subsection{Results}{
 \subsection{Gaussian}{

 r2m : \code{\link[MuMIn:r.squaredGLMM]{MuMIn::r.squaredGLMM}}

 r2c : \code{\link[MuMIn:r.squaredGLMM]{MuMIn::r.squaredGLMM}}

 AIC : \code{\link[stats:AIC]{stats::AIC}}

 AICc : \code{\link[MuMIn:AICc]{MuMIn::AICc}}

 BIC : \code{\link[stats:BIC]{stats::BIC}}

 }
 \subsection{Binomial}{

 Confusion matrix: \code{\link[caret:confusionMatrix]{caret::confusionMatrix}}

 ROC: \code{\link[pROC:roc]{pROC::roc}}

 MCC: \code{\link[mltools:mcc]{mltools::mcc}}
 }
 }
}
\examples{
# Attach packages
library(cvms)
library(groupdata2) # fold()
library(dplyr) # \%>\% arrange() mutate()

# Data is part of cvms
data <- participant.scores

# Set seed for reproducibility
set.seed(7)

# Fold data
data <- fold(data, k = 4,
             cat_col = 'diagnosis',
             id_col = 'participant') \%>\%
    mutate(diagnosis = as.factor(diagnosis)) \%>\%
    arrange(.folds)

# Cross-validate multiple formulas

formulas_gaussian <- c("score ~ diagnosis",
                       "score ~ age")
formulas_binomial <- c("diagnosis ~ score",
                       "diagnosis ~ age")

\donttest{
# Gaussian

# Create model function with args 'train_data' and 'formula'
# that returns a fitted model object
lm_model_fn <- function(train_data, formula){
    lm(formula = formula, data = train_data)
}

# Cross-validate the model function
cross_validate_fn(data,
                  model_fn = lm_model_fn,
                  formulas = formulas_gaussian,
                  type = 'gaussian',
                  fold_cols = ".folds")

# Binomial

# Create model function with args 'train_data' and 'formula'
# that returns a fitted model object
glm_model_fn <- function(train_data, formula){
    glm(formula = formula, data = train_data, family = "binomial")
}

# Cross-validate the model function
cross_validate_fn(data,
                  model_fn = glm_model_fn,
                  formulas = formulas_binomial,
                  type = 'binomial',
                  fold_cols = ".folds")

# Support Vector Machine (svm)

# Create model function with args 'train_data' and 'formula'
# that returns a fitted model object
svm_model_fn <- function(train_data, formula){
    e1071::svm(formula = formula,
               data = train_data,
               kernel = "linear",
               type = "C-classification")
}

# Cross-validate the model function
cross_validate_fn(data,
                  model_fn = svm_model_fn,
                  formulas = formulas_binomial,
                  type = 'binomial',
                  fold_cols = ".folds")

# Naive Bayes

# Create model function with args 'train_data' and 'formula'
# that returns a fitted model object
nb_model_fn <- function(train_data, formula){
    e1071::naiveBayes(formula = formula,
                      data = train_data)
}

# Create predict function with args 'test_data', 'model', and 'formula'
# that returns predictions in right format (here, a one-column matrix)
# Note the type = "raw" and that we pick the probabilities for class 1 with [,2]
nb_predict_fn <- function(test_data, model, formula = NULL){
  stats::predict(object = model, newdata = test_data,
                 type = "raw", allow.new.levels = TRUE)[,2]
}

# Cross-validate the model function
cross_validate_fn(data,
                  model_fn = nb_model_fn,
                  formulas = formulas_binomial,
                  type = 'binomial',
                  predict_fn = nb_predict_fn,
                  fold_cols = ".folds")

}
# Use parallelization

\donttest{
# Attach doParallel and register four cores
# Uncomment:
# library(doParallel)
# registerDoParallel(4)

# Create list of 20 model formulas
formulas <- rep(c("score~diagnosis",
                "score~age"), 10)

# Cross-validate a list of 20 model formulas in parallel
system.time({cross_validate_fn(data,
                               model_fn = lm_model_fn,
                               formulas = formulas,
                               type = 'gaussian',
                               fold_cols = ".folds",
                               parallel = TRUE)})

# Cross-validate a list of 20 model formulas sequentially
system.time({cross_validate_fn(data,
                               model_fn = lm_model_fn,
                               formulas = formulas,
                               type = 'gaussian',
                               fold_cols = ".folds",
                               parallel = FALSE)})
}
}
\seealso{
Other validation functions: \code{\link{cross_validate}},
  \code{\link{validate}}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}

Benjamin Hugh Zachariae
}
\concept{validation functions}
