% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_confusion_matrix.R
\name{plot_confusion_matrix}
\alias{plot_confusion_matrix}
\title{Plot a confusion matrix}
\usage{
plot_confusion_matrix(
  conf_matrix,
  targets_col = "Target",
  predictions_col = "Prediction",
  counts_col = "N",
  add_counts = TRUE,
  add_normalized = TRUE,
  add_row_percentages = TRUE,
  add_col_percentages = TRUE,
  add_arrows = TRUE,
  add_zero_shading = TRUE,
  counts_on_top = FALSE,
  palette = "Blues",
  theme_fn = ggplot2::theme_minimal,
  place_x_axis_above = TRUE,
  rotate_y_text = TRUE,
  digits = 1,
  font_counts = font(),
  font_normalized = font(),
  font_row_percentages = font(),
  font_col_percentages = font(),
  arrow_size = 0.048,
  arrow_nudge_from_text = 0.065,
  tile_border_color = NA,
  tile_border_size = 0.1,
  tile_border_linetype = "solid",
  darkness = 0.8
)
}
\arguments{
\item{conf_matrix}{Confusion matrix tibble with each combination of
targets and predictions along with their counts.

 E.g. for a binary classification:

 \tabular{rrr}{
  \strong{Target} \tab \strong{Prediction} \tab \strong{N} \cr
  class_1 \tab class_1 \tab 5 \cr
  class_1 \tab class_2 \tab 9 \cr
  class_2 \tab class_1 \tab 3 \cr
  class_2 \tab class_2 \tab 2 \cr
 }

 As created with the various evaluation functions in \code{cvms}, like
 \code{\link[cvms:evaluate]{evaluate()}}.

 \strong{Note}: If you supply the results from \code{\link[cvms:evaluate]{evaluate()}}
 or \code{\link[cvms:confusion_matrix]{confusion_matrix()}} directly,
 the confusion matrix tibble is extracted automatically, if possible.}

\item{targets_col}{Name of column with target levels.}

\item{predictions_col}{Name of column with prediction levels.}

\item{counts_col}{Name of column with a count for each combination
of the target and prediction levels.}

\item{add_counts}{Add the counts to the middle of the tiles. (Logical)}

\item{add_normalized}{Normalize the counts to percentages and
add to the middle of the tiles. (Logical)}

\item{add_row_percentages}{Add the row percentages,
 i.e. how big a part of its row the tile makes up. (Logical)

 By default, the row percentage is placed to the right of the tile, rotated 90 degrees.}

\item{add_col_percentages}{Add the column percentages,
 i.e. how big a part of its column the tile makes up. (Logical)

 By default, the row percentage is placed at the bottom of the tile.}

\item{add_arrows}{Add the arrows to the row and col percentages. (Logical)

 Note: Adding the arrows requires the \code{rsvg} and \code{ggimage} packages.}

\item{add_zero_shading}{Add image of skewed lines to zero-tiles. (Logical)

 Note: Adding the zero-shading requires the \code{rsvg} and \code{ggimage} packages.}

\item{counts_on_top}{Switch the counts and normalized counts,
such that the counts are on top. (Logical)}

\item{palette}{Color scheme. Passed directly to \code{palette} in
 \code{\link[ggplot2:scale_fill_distiller]{ggplot2::scale_fill_distiller}}.

 Try these palettes: \code{"Greens"}, \code{"Oranges"},
 \code{"Greys"}, \code{"Purples"}, \code{"Reds"},
 as well as the default \code{"Blues"}.}

\item{theme_fn}{The \code{ggplot2} theme function to apply.}

\item{place_x_axis_above}{Move the x-axis text to the top and reverse the levels such that
the "correct" diagonal goes from top left to bottom right. (Logical)}

\item{rotate_y_text}{Whether to rotate the y-axis text to
be vertical instead of horizontal. (Logical)}

\item{digits}{Number of digits to round to (percentages only).
 Set to a negative number for no rounding.

 Can be set for each font individually via the \code{font_*} arguments.}

\item{font_counts}{List of font settings for the counts.
Can be provided with \code{\link[cvms:font]{font()}}.}

\item{font_normalized}{List of font settings for the normalized counts.
Can be provided with \code{\link[cvms:font]{font()}}.}

\item{font_row_percentages}{List of font settings for the row percentages.
Can be provided with \code{\link[cvms:font]{font()}}.}

\item{font_col_percentages}{List of font settings for the column percentages.
Can be provided with \code{\link[cvms:font]{font()}}.}

\item{arrow_size}{Size of arrow icons. (Numeric)

 Is divided by \code{sqrt(nrow(conf_matrix))} and passed on
 to \code{\link[ggimage:geom_icon]{ggimage::geom_icon()}}.}

\item{arrow_nudge_from_text}{Distance from the percentage text to the arrow. (Numeric)}

\item{tile_border_color}{Color of the tile borders. Passed as \emph{\code{colour}} to
\code{\link[ggplot2:geom_tile]{ggplot2::geom_tile}}.}

\item{tile_border_size}{Size of the tile borders. Passed as \emph{\code{size}} to
\code{\link[ggplot2:geom_tile]{ggplot2::geom_tile}}.}

\item{tile_border_linetype}{Linetype for the tile borders. Passed as \emph{\code{linetype}} to
\code{\link[ggplot2:geom_tile]{ggplot2::geom_tile}}.}

\item{darkness}{How dark the darkest colors should be, between 0 and 1, where 1 is darkest.

 Technically, a lower value increases the upper limit in
 \code{\link[ggplot2:scale_fill_distiller]{ggplot2::scale_fill_distiller}}.}
}
\value{
A \code{ggplot2} object representing a confusion matrix. Color intensity depends on the counts.

 By default, each tile has the normalized count
 (overall percentage) and count in the middle, the
 column percentage at the bottom, and the
 row percentage to the right and rotated 90 degrees.

 In the "correct" diagonal (upper left to bottom right, by default),
 the column percentages are the class-level sensitivity scores,
 while the row percentages are the class-level positive predictive values.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

 Creates a \code{\link[ggplot2:ggplot]{ggplot2}} object representing a confusion matrix with counts,
 overall percentages, row percentages and column percentages.

 The confusion matrix can be created with \code{\link[cvms:evaluate]{evaluate()}}. See \code{Examples}.
}
\details{
Inspired by Antoine Sachet's answer at https://stackoverflow.com/a/53612391/11832955
}
\examples{
\donttest{
# Attach cvms
library(cvms)
library(ggplot2)

# Two classes

# Create targets and predictions data frame
data <- data.frame(
  "target" = c("A", "B", "A", "B", "A", "B",
               "A", "B", "A", "B", "A", "B"),
  "prediction" = c("B", "B", "A", "A", "A", "B",
                   "B", "B", "A", "B", "A", "A"),
  stringsAsFactors = FALSE
)

# Evaluate predictions and create confusion matrix
eval <- evaluate(
  data = data,
  target_col = "target",
  prediction_cols = "prediction",
  type = "binomial"
)

# Inspect confusion matrix tibble
eval[["Confusion Matrix"]][[1]]

# Plot confusion matrix
# Supply confusion matrix tibble directly
plot_confusion_matrix(eval[["Confusion Matrix"]][[1]])
# Plot first confusion matrix in evaluate() output
plot_confusion_matrix(eval)

# Three (or more) classes

# Create targets and predictions data frame
data <- data.frame(
  "target" = c("A", "B", "C", "B", "A", "B", "C",
               "B", "A", "B", "C", "B", "A"),
  "prediction" = c("C", "B", "A", "C", "A", "B", "B",
                   "C", "A", "B", "C", "A", "C"),
  stringsAsFactors = FALSE
)

# Evaluate predictions and create confusion matrix
eval <- evaluate(
  data = data,
  target_col = "target",
  prediction_cols = "prediction",
  type = "multinomial"
)

# Inspect confusion matrix tibble
eval[["Confusion Matrix"]][[1]]

# Plot confusion matrix
# Supply confusion matrix tibble directly
plot_confusion_matrix(eval[["Confusion Matrix"]][[1]])
# Plot first confusion matrix in evaluate() output
plot_confusion_matrix(eval)

# Counts only
plot_confusion_matrix(
  eval[["Confusion Matrix"]][[1]],
  add_normalized = FALSE,
  add_row_percentages = FALSE,
  add_col_percentages = FALSE
)

# Change color palette to green
# Change theme to \code{theme_light}.
plot_confusion_matrix(
  eval[["Confusion Matrix"]][[1]],
  palette = "Greens",
  theme_fn = ggplot2::theme_light
)

# The output is a ggplot2 object
# that you can add layers to
# Here we change the axis labels
plot_confusion_matrix(eval[["Confusion Matrix"]][[1]]) +
  ggplot2::labs(x = "True", y = "Guess")
}
}
\seealso{
Other plotting functions: 
\code{\link{font}()},
\code{\link{plot_metric_density}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{plotting functions}
