% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/validation.R
\name{validation_ui}
\alias{validation_ui}
\alias{validation_server}
\title{Validation module}
\usage{
validation_ui(id, display = c("dropdown", "inline"), max_height = NULL, ...)

validation_server(
  id,
  data,
  n_row = NULL,
  n_col = NULL,
  n_row_label = i18n("Valid number of rows"),
  n_col_label = i18n("Valid number of columns"),
  btn_label = i18n("Dataset validation:"),
  rules = NULL,
  bs_version = 3
)
}
\arguments{
\item{id}{Module's ID.}

\item{display}{Display validation results in a dropdown menu
by clicking on a button or display results directly in interface.}

\item{max_height}{Maximum height for validation results element, useful if you have many rules.}

\item{...}{Arguments passed to \code{actionButton} or \code{uiOutput} depending on display mode,
you cannot use \code{inputId}/\code{outputId}, \code{label} or \code{icon} (button only).}

\item{data}{a \code{reactive} function returning a \code{data.frame}.}

\item{n_row, n_col}{A one-sided formula to check number of rows and columns respectively, see below for examples.}

\item{n_row_label, n_col_label}{Text to be displayed with the result of the check for number of rows/columns.}

\item{btn_label}{Label for the dropdown button, will be followed by validation result.}

\item{rules}{An object of class \code{validator} created with \code{validate::validator}.}

\item{bs_version}{Bootstrap version used, it may affect rendering, especially status badges.}
}
\value{
\itemize{
\item UI: HTML tags that can be included in shiny's UI
\item Server: a \code{list} with two slots:
\itemize{
\item \strong{status}: a \code{reactive} function returning the best status available between \code{"OK"}, \code{"Failed"} or \code{"Error"}.
\item \strong{details}: a \code{reactive} function returning a \code{list} with validation details.
}
}
}
\description{
Check that a dataset respect some validation expectations.
}
\examples{
library(datamods)
library(shiny)

if (requireNamespace("validate")) {
  library(validate)

  # Define some rules to be applied to data
  myrules <- validator(
    is.character(Manufacturer) | is.factor(Manufacturer),
    is.numeric(Price),
    Price > 12, # we should use 0 for testing positivity, but that's for the example
    !is.na(Luggage.room),
    in_range(Cylinders, min = 4, max = 8),
    Man.trans.avail \%in\% c("Yes", "No")
  )
  # Add some labels
  label(myrules) <- c(
    "Variable Manufacturer must be character",
    "Variable Price must be numeric",
    "Variable Price must be strictly positive",
    "Luggage.room must not contain any missing values",
    "Cylinders must be between 4 and 8",
    "Man.trans.avail must be 'Yes' or 'No'"
  )
  # you can also add a description()

  ui <- fluidPage(
    tags$h2("Validation"),
    fluidRow(
      column(
        width = 4,
        radioButtons(
          inputId = "dataset",
          label = "Choose dataset:",
          choices = c("mtcars", "MASS::Cars93")
        ),
        tags$p("Dropdown example:"),
        validation_ui("validation1"),

        tags$br(),

        tags$p("Inline example:"),
        validation_ui("validation2", display = "inline")
      ),
      column(
        width = 8,
        tags$b("Status:"),
        verbatimTextOutput("status"),
        tags$b("Details:"),
        verbatimTextOutput("details")
      )
    )
  )

  server <- function(input, output, session) {

    dataset <- reactive({
      if (input$dataset == "mtcars") {
        mtcars
      } else {
        MASS::Cars93
      }
    })

    results <- validation_server(
      id = "validation1",
      data = dataset,
      n_row = ~ . > 20, # more than 20 rows
      n_col = ~ . >= 3, # at least 3 columns
      rules = myrules
    )

    validation_server(
      id = "validation2",
      data = dataset,
      n_row = ~ . > 20, # more than 20 rows
      n_col = ~ . >= 3, # at least 3 columns
      rules = myrules
    )

    output$status <- renderPrint(results$status())
    output$details <- renderPrint(results$details())

  }

  if (interactive())
    shinyApp(ui, server)
}
}
