% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_models.R
\name{fit_two_layer}
\alias{fit_two_layer}
\title{MCMC sampling for two layer deep GP}
\usage{
fit_two_layer(
  x,
  y,
  D = ifelse(is.matrix(x), ncol(x), 1),
  nmcmc = 10000,
  trace = TRUE,
  w_0 = suppressWarnings(matrix(x, nrow = length(y), ncol = D)),
  g_0 = 0.01,
  theta_y_0 = 0.5,
  theta_w_0 = 1,
  true_g = NULL
)
}
\arguments{
\item{x}{vector or matrix of input locations}

\item{y}{vector of response values}

\item{D}{integer designating dimension of hidden layer, defaults to dimension of \code{x}}

\item{nmcmc}{number of MCMC iterations}

\item{trace}{logical indicating whether to print iteration progress}

\item{w_0}{initial value for hidden layer \code{w}, defaults to identity mapping (must be 
matrix of dimension \code{nrow(x)} by \code{D} or dimension \code{nrow(x) - 1} by \code{D})}

\item{g_0}{initial value for \code{g}}

\item{theta_y_0}{initial value for \code{theta_y} (length scale of outer layer)}

\item{theta_w_0}{initial value for \code{theta_w} (length scale of inner layer), 
may be single value or vector of length \code{D}}

\item{true_g}{if true nugget is known it may be specified here (set to a small
value to make fit deterministic)}
}
\value{
a list of the S3 class "\code{dgp2}" with elements:
\itemize{
  \item \code{x}: copy of input matrix
  \item \code{y}: copy of response vector
  \item \code{nmcmc}: number of MCMC iterations
  \item \code{g}: vector of MCMC samples for \code{g}
  \item \code{theta_y}: vector of MCMC samples for \code{theta_y} (length
        scale of outer layer)
  \item \code{theta_w}: matrix of MCMC samples for \code{theta_w} (length 
        scale of inner layer)
  \item \code{w}: list of MCMC samples for hidden layer \code{w}
  \item \code{time}: computation time in seconds
}
}
\description{
Conducts MCMC sampling of hyperparameters and hidden layer 
    "\code{w}" for a two layer deep GP.  Covariance structure is based on 
    inverse exponentiated squared euclidean distance.  Separate length scale 
    parameters "\code{theta_w}" and "\code{theta_y}" govern the correlation 
    strength of the hidden layer and outer layer respectively.  Nugget 
    parameter "\code{g}" governs noise on the outer layer.
}
\details{
Maps inputs "\code{x}" through hidden layer "\code{w}" to outputs 
    "\code{y}".  Conducts sampling of the hidden layer using Elliptical Slice 
    sampling.  Utilizes Metropolis Hastings sampling of the length scale and
    nugget parameters with a  uniform proposal function (ranging from half 
    to twice the previous iteration) and the following priors:
    \itemize{
        \item \code{prior(g) <- dgamma(g, 1.5, 3.9)}
        \item \code{prior(theta_w) <- dgamma(theta_w, 1.5, 3.9/4)}
        \item \code{prior(theta_y) <- dgamma(theta_y, 1.5, 3.9/6)}
    }
    These priors are designed for "\code{x}" scaled to [0,1] and "\code{y}" 
    scaled to have mean zero and variance 1.  The output object of class 
    "\code{dgp2}" is designed for use with \code{continue}, \code{trim}, 
    and \code{predict}. If \code{w_0} is of dimension \code{nrow(x) - 1} by 
    \code{D}, the final row is predicted using kriging.  This is helpful in 
    sequential design when adding a new input location and starting the MCMC 
    at the place where the previous MCMC left off.
}
\examples{
# Toy example (runs in less than 5 seconds) -----------------------------------
# This example uses a small number of MCMC iterations in order to run quickly
# More iterations are required to get appropriate fits
# Function defaults are recommended (see additional example below)

f <- function(x) {
  if (x <= 0.4) return(-1)
  if (x >= 0.6) return(1)
  if (x > 0.4 & x < 0.6) return(10*(x-0.5))
}
x <- seq(0.05, 0.95, length = 7)
y <- sapply(x, f)
x_new <- seq(0, 1, length = 100)

# Fit model and calculate ALC
fit <- fit_two_layer(x, y, nmcmc = 500)
fit <- trim(fit, 400)
fit <- predict(fit, x_new)
alc <- ALC(fit)

\donttest{
# Two Layer and ALC -----------------------------------------------------------

f <- function(x) {
  exp(-10 * x) * (cos(10 * pi * x - 1) + sin(10 * pi * x - 1)) * 5 - 0.2
}

# Training data
x <- seq(0, 1, length = 30)
y <- f(x) + rnorm(30, 0, 0.05)

# Testing data
xx <- seq(0, 1, length = 100)
yy <- f(xx)

# Conduct MCMC
fit <- fit_two_layer(x, y, D = 1, nmcmc = 9000)
fit <- continue(fit, 1000)
plot(fit) # investigate trace plots
fit <- trim(fit, 8000, 2)

# Option 1 - calculate ALC from MCMC iterations
alc <- ALC(fit, xx)

# Option 2 - calculate ALC after predictions
fit <- predict(fit, xx)
alc <- ALC(fit)

# Visualize fit
plot(fit)
par(new = TRUE) # overlay ALC
plot(xx, alc$value, type = 'l', lty = 2, axes = FALSE, xlab = '', ylab = '')

# Select next design point
x_new <- xx[which.max(alc$value)]

# Evaluate fit
rmse(yy, fit$mean) # lower is better
score(yy, fit$mean, fit$Sigma) # higher is better
}

}
\references{
Damianou, A and N Lawrence. (2013). "Deep gaussian processes." 
    \emph{Artificial Intelligence and Statistics}, 207-215. \cr\cr
Murray, I, RP Adams, and D MacKay. 2010. "Elliptical slice sampling." 
    \emph{Journal of Machine Learning Research 9}, 541-548.
}
