% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detquery.R
\name{det.query}
\alias{det.query}
\title{Density Estimation based on Distribution Element Trees}
\usage{
det.query(det, x, cores = 0)
}
\arguments{
\item{det}{distribution element tree object resulting from \code{\link{det.construct}} based on data with \code{d} components or dimensions.}

\item{x}{matrix containing \code{n} query points (columns) with \code{d} components or dimensions (rows).}

\item{cores}{for large query-point sets, \code{cores > 1} allows for parallel tree query. The default is \code{cores = 0}, which allocates half of the available cores (see \code{\link[parallel]{detectCores}}). \code{cores = 1} corresponds to serial tree query.}
}
\value{
A vector containing the probability density at the query points \code{x} is returned.
}
\description{
The function \code{det.query} evaluates probability densities at the query points \code{x} based on a distribution element tree (DET). The latter is calculable with \code{\link{det.construct}} based on available data.
}
\examples{
## 1d example
require(stats); require(graphics)
# DET generation based on Gaussian data
det <- det.construct(matrix(rnorm(1e5,2,3), nrow = 1), cores = 2)
# density evaluation based on DET at equidistant query points
x <- matrix(seq(-10,14,0.01), nrow = 1); p <- det.query(det, x, cores = 1)
# compare DET estimate (black) against Gaussian reference (red)
plot(x, p, type = "l", col = "black")
lines(x, dnorm(x,2,3), col = "red")

## 2d example
require(stats); require(graphics)
# mean and covariance of Gaussian, data generation
mu <- c(3,5); C <- matrix(c(4.0,-2.28,-2.28,1.44), nrow = 2)
A <- eigen(C); B <- diag(A$values); A <- A$vectors
x <- matrix(rnorm(2e4), nrow = 2)
x <- t(A \%*\% (sqrt(B) \%*\% x) + mu \%*\% t(rep(1,dim(x)[2])))
# bounds and resolution of x1-x2 query grid
lb <- c(-5,0); ub <- c(11,10); np <- c(320,200)
x1 <- lb[1] + (ub[1]-lb[1])*((1:np[1])-0.5)/np[1]
x2 <- lb[2] + (ub[2]-lb[2])*((1:np[2])-0.5)/np[2]
xp <- rbind(rep(x1,np[2]), rep(x2,each = np[1])) # grid points
# plotting
split.screen(c(2, 2)); screen(1)
plot(x, type = "p", pch = ".", asp = 1, main = "data")
# DET estimator
det <- det.construct(t(x), cores = 1)
yd <- matrix(det.query(det, xp, cores = 2), nrow = np[1])
screen(2)
image(list(x = x1, y = x2, z = yd), asp = 1,
      col = grDevices::gray((100:0)/100), main = "det")
# Gaussian density for comparison
yr <- yr <- exp(-1/2 * colSums(
   (t(solve(C)) \%*\% (xp - mu\%*\%t(rep(1,dim(xp)[2])))) *
                    (xp - mu\%*\%t(rep(1,dim(xp)[2]))))
                              ) / sqrt((2*pi)^2*det(C))
yr <- matrix(yr, nrow = np[1])
screen(3)
image(list(x = x1, y = x2, z = yr), asp = 1,
      col = grDevices::gray((100:0)/100), main = "reference")
}
