\name{mta_next}
\alias{mta_next}
\alias{print.mta_next}
\title{MTA optimal dose determination}
\description{
\code{mta_next} is used to determine the next optimal dose to administer in a Phase I clinical trial for Molecularly Targeted Agent using the design proposed by Riviere et al. entitled "Phase I/II Dose-Finding Design for Molecularly Targeted Agent: Plateau Determination using Adaptive Randomization".
}
\usage{
mta_next(ngroups=1, group_cur=1, ndose, prior_tox, prior_eff, tox_max, eff_min, 
cohort_start, cohort, final=FALSE, method="MTA-RA", s_1=function(n_cur){0.2}, 
s_2=0.07,  group_pat, id_dose, toxicity, tite=TRUE, efficacy, time_follow, 
time_eff, time_full, cycle, c_tox=0.90, c_eff=0.40, seed = 1907)
}
\arguments{                
  \item{ngroups}{Number of groups for the dose-finding process leading to the recommendation of different dose levels. Several groups of efficacy (e.g. based on biomarker) sharing the same toxicity can be considered. The default value is set at 1.}
  \item{group_cur}{Group number for which the estimation and the optimal dose determination is required by the function. The default value is set at 1.}
  \item{ndose}{Number of dose levels.}
  \item{prior_tox}{A vector of initial guesses of toxicity probabilities associated with the doses. Must be of same length as \code{ndose}.}
  \item{prior_eff}{A vector of initial guesses of efficacy probabilities associated with the doses for \code{group_cur}. Must be of same length as \code{ndose}.}
  \item{tox_max}{Toxicity upper bound, i.e. maximum acceptable toxicity probability.}
  \item{eff_min}{Efficacy lower bound, i.e. minimum acceptable efficacy probability.}
  \item{cohort_start}{Cohort size for the start-up phase.}
  \item{cohort}{Cohort size for the model phase.}
  \item{final}{A boolean with value TRUE if the trial is finished and the recommended dose for further phases should be given, or FALSE (default value) if the dose determination is performed for the next cohort of patients.}
  \item{method}{A character string to specify the method for dose allocation (<=> plateau determination). The default method "MTA-RA" use adaptive randomization on posterior probabilities for the plateau location. Method based on difference in efficacy probabilities is specified by "MTA-PM".}
  \item{s_1}{A function depending on the number of patients included used for adaptive randomization in plateau determination, only used if the estimation method chosen is "MTA-RA". The default function is function(n_cur,n){0.2}.}
  \item{s_2}{Cutoff value for plateau determination, only used if the estimation method chosen is "MTA-PM". Can be seen as the minimal efficacy difference of practical importance. The default value is 0.07.}
  \item{group_pat}{A vector indicating the group number associated with each patient included in the trial.}
  \item{id_dose}{A vector indicating the dose levels administered to each patient included in the trial. Must be of same length as \code{group_pat}.}
  \item{toxicity}{A vector of observed toxicities (DLTs) for each patient included in the trial. Must be of same length as \code{group_pat}.}
  \item{tite}{A boolean indicating if the efficacy is considered as a time-to-event (default value TRUE), or if it is a binary outcome (FALSE).}
  \item{efficacy}{A vector of observed efficacies for each patient included in the trial. Must be of same length as \code{group_pat}. This argument is used/required only if tite=FALSE. The observed efficacies of patients belonging to other groups than \code{group_cur} should also be filled (although not used) in the same order as \code{group_pat} (NA can be put).}
  \item{time_follow}{A vector of follow-up times for each patient included in the trial. Must be of same length as \code{group_pat}. This argument is used/required only if tite=TRUE.}
  \item{time_eff}{A vector of times-to-efficacy for each patient included in the trial. If no efficacy was observed for a patient, must be filled with +Inf. Must be of same length as \code{group_pat}. This argument is used/required only if tite=TRUE.}
  \item{time_full}{Full follow-up time window. This argument is used only if tite=TRUE.}
  \item{cycle}{Minimum waiting time between two dose cohorts (usually a toxicity cycle). This argument is used only if tite=TRUE.}
  \item{c_tox}{Tocixity threshold for decision rules. The default value is set at 0.90.}
  \item{c_eff}{Efficacy threshold for decision rules. The default value is set at 0.40.}
  \item{seed}{Seed of the random number generator. Default value is set at 1907.}
}


\value{
An object of class "mta_next" is returned, consisting of determination of the next optimal dose level to administer and estimations. Objects generated by \code{mta_next} contain at least the following components: 

\item{prior_tox}{Prior toxicities.}
\item{prior_eff}{Prior efficacies.}
\item{pat_incl_group}{Number of patients included.}
\item{}{Number of observed toxicities.}
\item{pi}{Estimated toxicity probabilities (if the start-up ended).}
\item{ptox_inf}{Estimated probabilities that the toxicity probability is inferior to \code{tox_max} (if the start-up ended).}
\item{}{Number of observed efficacies.}
\item{resp}{Estimated efficacy probabilities (if the start-up ended).}
\item{1-qeff_inf}{Estimated probabilities that the efficacy probability is superior to \code{eff_min} (if the start-up ended).}         
\item{proba_tau}{Posterior probabilities for the plateau location.}
\item{group_cur}{Current Group for dose determination.}
\item{startup_end}{Start-up phase is ended or not.}
\item{cdose}{NEXT RECOMMENDED DOSE.}
\item{ngroups}{Number of groups.}
\item{final}{Maximim sample size reached.}
\item{}{Allocation method.}
\item{tox_max}{Toxicity upper bound (if the start-up ended).}
\item{eff_min}{Efficacy lower bound (if the start-up ended).}
\item{c_tox}{Toxicity threshold (if the start-up ended).}
\item{c_eff}{Efficacy threshold (if the start-up ended).}
\item{tite}{Type of outcome for efficacy (time-to-event or binary).}
\item{time_full}{If efficacy is a time-to-event, full follow-up time is also reminded.}
\item{cycle}{If efficacy is a time-to-event, minimum waiting time between two dose cohorts (cycle) is also reminded.}
}
\references{
Riviere, M-K., Yuan, Y., Jourdan, J-H., Dubois, F., and Zohar, S. (2014). Phase I/II Dose-Finding Design for Molecularly Targeted Agent: Plateau Determination using Adaptive Randomization. Statistics in medicine.
}
\note{The "MTA-PM" method is not implemented for non-binary efficacy, as "MTA-RA" is recommended for general use.}
\author{Jacques-Henri Jourdan and Marie-Karelle Riviere-Jourdan \email{marie-karelle.riviere@inserm.fr}}
\seealso{
  \code{\link{mta_sim}}.
}
\examples{
prior_tox = c(0.02, 0.06, 0.12, 0.20, 0.30, 0.40)
prior_eff = c(0.12, 0.20, 0.30, 0.40, 0.50, 0.59)
group_pat_1 = rep(1,33)
id_dose_1 = c(1,1,1,2,2,2,3,3,3,4,4,4,5,5,5,4,4,4,5,5,5,6,6,6,3,3,3,4,4,4,3,3,3)
tox_1 = c(0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,1,1,0,0,0,0,0,1,0,0,0,0)
time_follow_1 = c(rep(7,30),6.8,5,3.5)
time_eff_1 = c(rep(+Inf,8),4,+Inf,+Inf,+Inf,3,6,+Inf,+Inf,2,+Inf,+Inf,4.5,+Inf,
+Inf,3.2,+Inf,+Inf,2.4,6.1,+Inf,5.8,+Inf,+Inf,2.1,3.6) 
eff_2 = c(0,0,0,0,0,0,0,0,1,0,0,0,1,1,0,0,1,0,0,1,0,0,1,0,0,1,1,0,1,0,0,1,1)
group_pat_3 = c(1,2,3,2,1,2,3,1,2,3,3,2,2,1,3,1,2,3,1,2,3,3,3,2,1,1,2,1,2,2)
id_dose_3 = c(1,1,1,1,1,1,1,1,2,1,2,2,2,2,2,2,3,2,2,3,3,3,3,3,1,1,2,1,2,2)
toxicity_3 = c(0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,1,0,0,1,0,0,0,0,0,0,0)
efficacy_3 = c(NA,0,NA,0,NA,1,NA,NA,0,NA,NA,1,0,NA,NA,NA,0,NA,NA,1,NA,NA,NA,
0,NA,NA,0,NA,1,1)
s_1=function(n_cur){0.2*(1-n_cur/60)}

# One group, time-to-event
#mta1 = mta_next(ngroups=1, group_cur=1, ndose=6, prior_tox=prior_tox, 
#       prior_eff=prior_eff, tox_max=0.35, eff_min=0.20, cohort_start=3, 
#       cohort=3, method="MTA-PM", group_pat=group_pat_1, id_dose=id_dose_1, 
#       toxicity=tox_1, tite=TRUE, time_follow=time_follow_1, 
#       time_eff=time_eff_1, time_full=7, cycle=3, c_tox=0.90, c_eff=0.40)
#mta1
   
# One group, binary                
#mta2 = mta_next(ngroups=1, group_cur=1, ndose=6, prior_tox=prior_tox, 
#       prior_eff=prior_eff, tox_max=0.35, eff_min=0.20, cohort_start=3, 
#       cohort=3, final = TRUE, method="MTA-RA", group_pat=group_pat_1, 
#       id_dose=id_dose_1, toxicity=tox_1, tite=FALSE, efficacy=eff_2, 
#       seed = 190714)
#mta2

# Three groups, binary
#mta3 = mta_next(ngroups=3, group_cur=2, ndose=6, prior_tox=prior_tox, 
#       prior_eff=prior_eff, tox_max=0.35, eff_min=0.20, cohort_start=3, 
#       cohort=3, final = FALSE, s_1=s_1, group_pat=group_pat_3, 
#       id_dose=id_dose_3, toxicity=toxicity_3, tite=FALSE, efficacy=efficacy_3)
#mta3
}
