% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/diversityDynamics.r
\name{divDyn}
\alias{divDyn}
\title{Time series from metrics of diversity dynamics}
\usage{
divDyn(dat, tax, bin, ages = FALSE, breaks = NULL,
  coll = "collection_no", ref = "reference_no", om = NULL,
  noNAStart = FALSE, data.frame = TRUE, filterNA = FALSE)
}
\arguments{
\item{dat}{\code{(data.frame)} Fossil occurrence table.}

\item{tax}{\code{(character)} Variable name of the occurring taxa (variable type: \code{factor} or \code{character} - such as \code{"genus"}}

\item{bin}{\code{(character)} Variable name of the bin numbers of the occurrences. This variable should be \code{numeric}.}

\item{ages}{\code{(logical)} Argument for setting the direction of time in \code{bin}. The default setting, \code{FALSE} will execute the function with higher numbers corresponding to later intervals. \code{TRUE} will reverse the direction of values in the \code{bins} variable, making originations extinctions and vice versa (use this for age input!, e.g 400 Ma, see examples).}

\item{breaks}{\code{(numeric)} If \code{NULL} (default) the used values in the \code{bin} variable will designate independent time slices that follow each other in succession. If a vector is provided, than the numeric entries in \code{bin} will be binned similarly to the \code{\link[graphics]{hist}} or \code{\link[base]{cut}} function. The order of elements in this vector is arbitrary.}

\item{coll}{\code{(character)} The variable name of the collection identifiers. (optional, only for use with the internal \code{\link{omit}} function)}

\item{ref}{\code{(character)} The variable name of the reference identifiers. (optional, only for use with the internal \code{\link{omit}} function)}

\item{om}{\code{(character)} The \code{om} argument of the \code{omit()} function. If set to \code{NULL} (default), then no occurrences will be omitted before the execution of the function.}

\item{noNAStart}{(logical) Useful when the entries in the \code{bin} variable do not start from bin no. 1, but positive integer bin numbers are provided. Then \code{noNAStart=TRUE} will cut the first part of the resulting table, so the first row will contain the estimates for the lowest bin number. In case of positive integer bin identifiers, and if \code{noNAStart=FALSE}, the index of the row will be the bin number.}

\item{data.frame}{\code{(logical)} Should the output be a \code{data.frame} or a \code{matrix}?}

\item{filterNA}{\code{(logical)} The \code{filterNA} parameter of the \code{\link{omit}} function.}
}
\description{
This function calculates various metrics from occurrence datasets in the form of time series.
}
\details{
The following variables are produced:

\code{bin}: Bin number, or the numeric identifier of the bin.

\code{tThrough}: Number of through-ranging taxa, taxa that have first occurrences before, and last occurrences after the focal bin.

\code{tOri}: Number of originating taxa, taxa that have first occurrences in the focal bin, and last occurrences after.

\code{tExt}: Number of taxa getting extinct. These are taxa that have first occurrences before the focal bin, and last occurrences after it.

\code{tSing}: Number of stratigraphic singleton (single-interval) taxa, taxa that only occur in the focal bin.

\code{t2d}: Number of lower two timers (Alroy, 2008; 2014), taxa that are present in the \emph{i}-1th and the ith bin (focal bin).

\code{t2u}: Number of upper two timers (Alroy, 2008; 2014), taxa that are present in the \emph{i}th (focal) and the \emph{i}+1th bin. (Alroy, 2008; 2014)

\code{tGFu}: Number of upper gap-fillers (Alroy, 2014), taxa that occurr in bin \emph{i}+2 and \emph{i}-1, but were not found in \emph{i}+1. (Alroy, 2014)

\code{tGFd}: Number of lower gap-fillers (Alroy, 2014), taxa that occurr in bin \emph{i}-2 and \emph{i}+1, but were not found in \emph{i}-1. (Alroy, 2014)

\code{t3}: Number of three timer taxa (Alroy, 2008; 2014), present in bin \emph{i}-1, \emph{i}, and \emph{i}+1. (Alroy, 2008; 2014)

\code{tPart}: Part timer taxa (Alroy, 2008; 2014), present in bin \emph{i}-1,and \emph{i}+1, but not in bin \emph{i}.

\code{extProp}: Proportional extinctions including single-interval taxa: \emph{(tExt + tSing) / (tThrough + tOri + tExt + tSing)}.

\code{oriProp}: Proportional originations including single-interval taxa:  \emph{(tOri + tSing) / (tThrough + tOri + tExt + tSing)}.

\code{extPC}: Per capita extinction rates of Foote (1999). \emph{-log(tExt/(tExt + tThrough))}.  Values are not normalized with bin lengths. Similar equations were used by Alroy (1996) but without taking the logarithm.

\code{oriPC}: Per capita origination rates of Foote (1999). \emph{-log(tOri/(tOri + tThrough))}. Values are not normalized with bin lengths. Similar equations were used by Alroy (1996) but without taking the logarithm.

\code{ext3t}: Three-timer extinction rates of Alroy (2008). \emph{log(t2d/t3)}.

\code{ori3t}: Three-timer origination rates of Alroy (2008). \emph{log(t2u/t3)}.

\code{extC3t}: Corrected three-timer extinction rates of Alroy (2008). \emph{ext3t[\emph{i}] + log(samp3t[\emph{i}+1])}.

\code{oriC3t}: Corrected three-timer origination rates of Alroy (2008). \emph{ori3t[\emph{i}] + log(samp3t[\emph{i}-1])}.

\code{divSIB}: Sampled-in-bin diversity (richness), the number of genera sampled in the focal bin.

\code{divCSIB}: Corrected sampled-in-bin diversity (richness). \emph{divSIB/samp3t*totSamp3t}, where \emph{totSamp3t} is total three-timer sampling completeness of the dataset (Alroy, 2008).

\code{divBC}: Boundary-crosser diversity (richness), the number of taxa with ranges crossing the boundaries of the interval. \emph{tExt + tOri + tThrough}.

\code{divRT}: Range-through diversity (richness), all taxa in the interval, based on the range-through assumption. \emph{(tSing + tOri + tExt + tThrough)}.

\code{sampRange}: Range-based sampling probability (Foote), \emph{(divSIB - tExt - tOri- t-Sing)/tThrough}

\code{samp3t}: Three-timer sampling completeness of Alroy (2008). \emph{t3/(t3+tPart)}

\code{extGF}: Gap-filler extinction rates of Alroy(2014). \emph{log((t2u + tPart)/(t3+tPart+tGFd))}

\code{oriGF}: Gap-filler origination rates of Alroy(2014). \emph{log((t2u + tPart)/(t3+tPart+tGFd))}

\code{E2f3}: Second-for-third extinction propotions of Alroy (2015). As these metrics are based on an algorithmic approach, for the equations please refer to the Alroy (2015, p. 634, right column and Eq. 4)). See source code (\url{http://www.github.com/adamkocsis/divDyn}) for the exact implementation, found in the \code{Metrics} function in the diversityDynamics.R file.

\code{O2f3}: Second-for-third origination propotions of Alroy (2015). Please see \code{E2f3}.

\code{ext2f3}: Second-for-third extinction rates (based on Alroy, 2015). Transformed to the usual rate form with \emph{log(1/(1-E2f3))}.

\code{ori2f3}: Second-for-third origination rates (based on Alroy, 2015). Transformed to the usual rate form with \emph{log(1/(1-O2f3))}.

\strong{References:}

Foote, M. (1999) Morphological Diversity In The Evolutionary Radiation Of Paleozoic and Post-Paleozoic Crinoids. Paleobiology 25, 1–115. doi:10.1017/S0094837300020236.

Alroy, J. (2008) Dynamics of origination and extinction in the marine fossil record. Proceedings of the National Academy of Science 105, 11536-11542. doi: 10.1073/pnas.0802597105

Alroy, J. (2014) Accurate and precise estimates of origination and extinction rates. Paleobiology 40, 374-397. doi: 10.1666/13036

Alroy, J. (2015) A more precise speciation and extinction rate estimator. Paleobiology 41, 633-639. doi: 10.1017/pab.2015.26
}
\examples{
# import data
  data(corals)
  data(stages)

# calculate metrics of diversity dynamics
   dd <- divDyn(corals, tax="genus", bin="stg")

# plotting
  tsplot(stages, shading="series", boxes="sys", xlim=c(260,0), 
    ylab="range-through diversity (genera)", ylim=c(0,230))
  lines(stages$mid, dd$divRT, lwd=2)

 # with omission of single reference taxa  
   ddNoSing <- divDyn(corals, tax="genus", bin="stg", om="ref")
   lines(stages$mid, ddNoSing$divRT, lwd=2, col="red")

 # using the estimated ages (less robust) - 10 million years
   # mean ages
   corals$me_ma <- apply(corals[, c("max_ma", "min_ma")], 1, mean)
   # ages reverse the direction of time! set ages to TRUE in this case
   ddRadio10 <- divDyn(corals, tax="genus", bin="me_ma", 
	breaks=seq(250,0,-10), ages=TRUE)
   lines(ddRadio10$bin, ddRadio10$divRT, lwd=2, col="green")
      
 # legend
   legend("topleft", legend=c("all", "no single-ref. taxa", "all, estimated ages"), 
     col=c("black", "red", "green"), lwd=c(2,2,2), bg="white")
   

}
