#' @title Function \code{make}
#' @description Run your project (build the targets).
#' @seealso \code{\link{workplan}}, \code{\link{workplan}},
#' \code{\link{backend}}, \code{\link{plot_graph}},
#' \code{\link{max_useful_jobs}}, \code{\link{shell_file}},
#' \code{\link{silencer_hook}}
#' @export
#'
#' @param plan workflow plan data frame.
#' A workflow plan data frame is a data frame
#' with a \code{target} column and a \code{command} column.
#' Targets are the objects and files that drake generates,
#' and commands are the pieces of R code that produce them.
#' Use the function \code{\link{workplan}()} to generate workflow plan
#' data frames easily, and see functions \code{\link{analyses}()},
#' \code{\link{summaries}()}, \code{\link{evaluate}()},
#' \code{\link{expand}()}, and \code{\link{gather}()} for
#' easy ways to generate large workflow plan data frames.
#'
#' @param targets character string, names of targets to build.
#' Dependencies are built too.
#'
#' @param envir environment to use. Defaults to the current
#' workspace, so you should not need to worry about this
#' most of the time. A deep copy of \code{envir} is made,
#' so you don't need to worry about your workspace being modified
#' by \code{make}. The deep copy inherits from the global environment.
#' Wherever necessary, objects and functions are imported
#' from \code{envir} and the global environment and
#' then reproducibly tracked as dependencies.
#'
#' @param verbose logical, whether to print progress to the console.
#' Skipped objects are not printed.
#'
#' @param hook function with at least one argument.
#' The hook is as a wrapper around the code that drake uses
#' to build a target (see the body of \code{drake:::build()}).
#' Hooks can control the side effects of build behavior.
#' For example, to redirect output and error messages to text files,
#' you might use the built-in \code{\link{silencer_hook}()}, as in
#' \code{make(my_plan, hook = silencer_hook)}.
#' The silencer hook is useful for distributed parallelism,
#' where the calling R process does not have control over all the
#' error and output streams. See also \code{\link{output_sink_hook}()}
#' and \code{\link{message_sink_hook}()}.
#' For your own custom hooks, treat the first argument as the code
#' that builds a target, and make sure this argument is actually evaluated.
#' Otherwise, the code will not run and none of your targets will build.
#' For example, \code{function(code){force(code)}} is a good hook
#' and \code{function(code){cat("Avoiding the code")}} is a bad hook.
#'
#' @param imports_only logical, whether to skip building the targets
#' in \code{plan} and just import objects and files.
#'
#' @param parallelism character, type of parallelism to use.
#' To list the options, call \code{\link{parallelism_choices}()}.
#' For detailed explanations, see \code{?\link{parallelism_choices}},
#' the tutorial vignettes, or the tutorial files generated by
#' \code{\link{example_drake}("basic")}
#'
#' @param jobs number of parallel processes or jobs to run.
#' For \code{"future_lapply"} parallelism, \code{jobs}
#' only applies to the imports.
#' See \code{future::future.options} for environment variables that
#' control the number of \code{future_lapply()} jobs for building targets.
#' For example, you might use \code{options(mc.cores = max_jobs)}.
#' See \code{\link{max_useful_jobs}()} or \code{\link{plot_graph}()}
#' to help figure out what the number of jobs should be.
#' Windows users should not set \code{jobs > 1} if
#' \code{parallelism} is \code{"mclapply"} because
#' \code{\link{mclapply}()} is based on forking. Windows users
#' who use \code{parallelism == "Makefile"} will need to
#' download and install Rtools.
#'
#' If \code{parallelism} is \code{"Makefile"},  Makefile-level parallelism is
#' only used for targets in your workflow plan data frame, not imports.  To
#' process imported objects and files, drake selects the best parallel backend
#' for your system and uses the number of jobs you give to the \code{jobs}
#' argument to \code{\link{make}()}. To use at most 2 jobs for imports and at
#' most 4 jobs for targets, run
#' \code{make(..., parallelism = "Makefile", jobs = 2, args = "--jobs=4")}
#'
#' @param packages character vector packages to load, in the order
#' they should be loaded. Defaults to \code{rev(.packages())}, so you
#' should not usually need to set this manually. Just call
#' \code{\link{library}()} to load your packages before \code{make()}.
#' However, sometimes packages need to be strictly forced to load
#' in a certain order, especially if \code{parallelism} is
#' \code{"Makefile"}. To do this, do not use \code{\link{library}()}
#' or \code{\link{require}()} or \code{\link{loadNamespace}()} or
#' \code{\link{attachNamespace}()} to load any libraries beforehand.
#' Just list your packages in the \code{packages} argument in the order
#' you want them to be loaded.
#' If \code{parallelism} is \code{"mclapply"},
#' the necessary packages
#' are loaded once before any targets are built. If \code{parallelism} is
#' \code{"Makefile"}, the necessary packages are loaded once on
#' initialization and then once again for each target right
#' before that target is built.
#'
#' @param prework character vector of lines of code to run
#' before build time. This code can be used to
#' load packages, set options, etc., although the packages in the
#' \code{packages} argument are loaded before any prework is done.
#' If \code{parallelism} is \code{"mclapply"}, the \code{prework}
#' is run once before any targets are built. If \code{parallelism} is
#' \code{"Makefile"}, the prework is run once on initialization
#' and then once again for each target right before that target is built.
#'
#' @param prepend lines to prepend to the Makefile if \code{parallelism}
#' is \code{"Makefile"}. See the vignettes
#' (\code{vignette(package = "drake")})
#' to learn how to use \code{prepend}
#' to take advantage of multiple nodes of a supercomputer.
#'
#' @param command character scalar, command to call the Makefile
#' generated for distributed computing.
#' Only applies when \code{parallelism} is \code{"Makefile"}.
#' Defaults to the usual \code{"make"}
#' (\code{\link{default_Makefile_command}()}),
#' but it could also be
#' \code{"lsmake"} on supporting systems, for example.
#' \code{command} and \code{args} are executed via
#' \code{\link{system2}(command, args)} to run the Makefile.
#' If \code{args} has something like \code{"--jobs=2"}, or if
#' \code{jobs >= 2} and \code{args} is left alone, targets
#' will be distributed over independent parallel R sessions
#' wherever possible.
#'
#' @param args command line arguments to call the Makefile for
#' distributed computing. For advanced users only. If set,
#' \code{jobs} and \code{verbose} are overwritten as they apply to the
#' Makefile.
#' \code{command} and \code{args} are executed via
#' \code{\link{system2}(command, args)} to run the Makefile.
#' If \code{args} has something like \code{"--jobs=2"}, or if
#' \code{jobs >= 2} and \code{args} is left alone, targets
#' will be distributed over independent parallel R sessions
#' wherever possible.
#'
#' @param recipe_command Character scalar, command for the
#' Makefile recipe for each target.
#'
#' @param clear_progress logical, whether to clear the saved record of
#' progress seen by \code{\link{progress}()} and \code{\link{in_progress}()}
#' before anything is imported or built.
#'
#' @param cache drake cache as created by \code{\link{new_cache}()}.
#' See also \code{\link{get_cache}()}, \code{\link{this_cache}()},
#' and \code{\link{recover_cache}()}
#'
#' @param timeout Seconds of overall time to allow before imposing
#' a timeout on a target. Passed to \code{R.utils::withTimeout()}.
#'
#' @param cpu Seconds of cpu time to allow before imposing
#' a timeout on a target. Passed to \code{R.utils::withTimeout()}.
#'
#' @param elapsed Seconds of elapsed time to allow before imposing
#' a timeout on a target. Passed to \code{R.utils::withTimeout()}.
#'
#' @param retries Number of retries to execute if the target fails.
#'
#' @param return_config logical, whether to return the internal list
#' of runtime configuration parameters used by \code{make()}.
#' This argument is deprecated. Now, a configuration list
#' is always invisibly returned.
#'
#' @examples
#' \dontrun{
#' load_basic_example()
#' outdated(my_plan) # Which targets need to be (re)built?
#' my_jobs = max_useful_jobs(my_plan) # Depends on what is up to date.
#' make(my_plan, jobs = my_jobs) # Build what needs to be built.
#' outdated(my_plan) # Everything is up to date.
#' reg2 = function(d){ # Change one of your functions.
#'   d$x3 = d$x^3
#'   lm(y ~ x3, data = d)
#' }
#' outdated(my_plan) # Some targets depend on reg2().
#' plot_graph(my_plan) # See how they fit in an interactive graph.
#' make(my_plan) # Rebuild just the outdated targets.
#' outdated(my_plan) # Everything is up to date again.
#' plot_graph(my_plan) # The colors changed in the graph.
#' clean() # Start from scratch
#' make(my_plan, parallelism = "Makefile", jobs = 4)
#' clean()
#' make(my_plan, parallelism = "Makefile", jobs = 4,
#'   recipe_command = "R -q -e")
#' }
make <- function(
  plan = workplan(),
  targets = drake::possible_targets(plan),
  envir = parent.frame(),
  verbose = TRUE,
  hook = function(code){
    force(code)
  },
  cache = drake::get_cache(verbose = verbose),
  parallelism = drake::default_parallelism(),
  jobs = 1,
  packages = rev(.packages()),
  prework = character(0),
  prepend = character(0),
  command = drake::default_Makefile_command(),
  args = drake::default_Makefile_args(
    jobs = jobs,
    verbose = verbose
  ),
  recipe_command = drake::default_recipe_command(),
  clear_progress = NULL,
  imports_only = FALSE,
  timeout = Inf,
  cpu = timeout,
  elapsed = timeout,
  retries = 0,
  return_config = NULL
){
  force(envir)

  if (!is.null(return_config)){
    warning(
      "The return_config argument to make() is deprecated. ",
      "Now, an internal configuration list is always invisibly returned.",
      call. = FALSE
    )
  }
  if (!is.null(clear_progress)){
    warning(
      "The clear_progress argument to make() is deprecated. ",
      "Progress is always cleared.",
      call. = FALSE
    )
  }

  config <- config(
    plan = plan,
    targets = targets,
    envir = envir,
    verbose = verbose,
    hook = hook,
    parallelism = parallelism,
    jobs = jobs,
    packages = packages,
    prework = prework,
    prepend = prepend,
    command = command,
    args = args,
    recipe_command = recipe_command,
    clear_progress = TRUE,
    cache = cache,
    timeout = timeout,
    cpu = cpu,
    elapsed = elapsed,
    retries = retries
  )
  check_config(config = config)
  store_config(config = config)
  initialize_session(config = config)
  if (imports_only){
    make_imports(config = config)
    return(invisible(config))
  }
  run_parallel_backend(config = config)
  console_up_to_date(config = config)
  return(invisible(config))
}

#' @title Function make_imports
#' @description just make the imports
#' @export
#' @seealso \code{\link{make}}, \code{\link{config}}
#' @param config a configuration list returned by \code{\link{config}()}
#' @examples
#' \dontrun{
#' load_basic_example()
#' con <- config(my_plan)
#' make_imports(config = con)
#' }
make_imports <- function(config){
  delete_these <- intersect(config$plan$target, V(config$graph)$name)
  config$graph <- delete_vertices(config$graph, v = delete_these)
  config$parallelism <- use_default_parallelism(config$parallelism)
  run_parallel_backend(config = config)
  invisible()
}

initialize_session <- function(config){
  config$cache$clear(namespace = "target_attempts")
  config$cache$set(
    key = "sessionInfo",
    value = sessionInfo(),
    namespace = "session"
  )
}
