#' @title Run your project (build the outdated targets).
#' \lifecycle{maturing}
#' @description This is the central, most important function
#' of the drake package. It runs all the steps of your
#' workflow in the correct order, skipping any work
#' that is already up to date.
#' See <https://github.com/ropensci/drake/blob/master/README.md#documentation>
#' for an overview of the documentation.
#' @section Interactive mode:
#' In interactive sessions, consider [r_make()], [r_outdated()], etc.
#' rather than [make()], [outdated()], etc. The `r_*()` `drake` functions
#' are more reproducible when the session is interactive.
#'
#' A serious drake workflow should be consistent and reliable,
#' ideally with the help of a master R script.
#' This script should begin in a fresh R session,
#' load your packages and functions in a dependable manner,
#' and then run `make()`. Example:
#' <https://github.com/wlandau/drake-examples/tree/master/gsp>.
#' Batch mode, especially within a container, is particularly helpful.
#'
#' Interactive R sessions are still useful,
#' but they easily grow stale.
#' Targets can falsely invalidate if you accidentally change
#' a function or data object in your environment.
#'
#' @section Self-invalidation:
#' It is possible to construct a workflow that tries to invalidate itself.
#' Example:
#' ```r
#' plan <- drake_plan(
#'   x = {
#'     data(mtcars)
#'     mtcars$mpg
#'   },
#'   y = mean(x)
#' )
#' ```
#' Here, because `data()` loads `mtcars` into the global environment,
#' the very act of building `x` changes the dependencies of `x`.
#' In other words, without safeguards, `x` would not be up to date at
#' the end of `make(plan)`.
#' Please try to avoid workflows that modify the global environment.
#' Functions such as `data()` belong in your setup scripts
#' prior to `make()`, not in any functions or commands that get called
#' during `make()` itself.
#'
#' For each target that is still problematic  (e.g.
#' <https://github.com/rstudio/gt/issues/297>)
#' you can safely run the command in its own special `callr::r()` process.
#' Example: <https://github.com/rstudio/gt/issues/297#issuecomment-497778735>. # nolint
#'
#' If that fails, you can run `make(plan, lock_envir = FALSE)`
#' to suppress environment-locking for all targets.
#' However, this is not usually recommended.
#' There are legitimate use cases for `lock_envir = FALSE`
#' (example: <https://ropenscilabs.github.io/drake-manual/hpc.html#parallel-computing-within-targets>) # nolint
#' but most workflows should stick with the default `lock_envir = TRUE`.
#'
#' @seealso
#'   [drake_plan()],
#'   [drake_config()],
#'   [vis_drake_graph()],
#'   [outdated()]
#' @export
#' @return nothing
#' @inheritParams drake_config
#' @param config A list generated by [drake_config()].
#'   If you supply `config` to `make()`, you must not supply any
#'   additional arguments.
#'
#'   [drake_config()] collects and sanitizes the multitude of
#'   parameters and settings that [make()] needs to do its job:
#'   the plan, packages,
#'   the environment of functions and initial data objects,
#'   parallel computing instructions,
#'   verbosity level, etc. Once you generate a list with
#'   [drake_config()], do not modify it by hand.
#'
#'   Utility functions such as [outdated()],
#'   [vis_drake_graph()], and [predict_runtime()]
#'   require output from
#'   [drake_config()] for the `config` argument.
#' @examples
#' \dontrun{
#' isolate_example("Quarantine side effects.", {
#' if (suppressWarnings(require("knitr"))) {
#' load_mtcars_example() # Get the code with drake_example("mtcars").
#' config <- drake_config(my_plan)
#' outdated(config) # Which targets need to be (re)built?
#' make(my_plan) # Build what needs to be built.
#' outdated(config) # Everything is up to date.
#' # Change one of your imported function dependencies.
#' reg2 = function(d) {
#'   d$x3 = d$x^3
#'   lm(y ~ x3, data = d)
#' }
#' outdated(config) # Some targets depend on reg2().
#' make(my_plan) # Rebuild just the outdated targets.
#' outdated(config) # Everything is up to date again.
#' if (requireNamespace("visNetwork", quietly = TRUE)) {
#' vis_drake_graph(config) # See how they fit in an interactive graph.
#' make(my_plan, cache_log_file = TRUE) # Write a CSV log file this time.
#' vis_drake_graph(config) # The colors changed in the graph.
#' # Run targets in parallel:
#' # options(clustermq.scheduler = "multicore") # nolint
#' # make(my_plan, parallelism = "clustermq", jobs = 2) # nolint
#' }
#' clean() # Start from scratch next time around.
#' }
#' })
#' }
make <- function(
  plan,
  targets = NULL,
  envir = parent.frame(),
  verbose = 1L,
  hook = NULL,
  cache = drake::drake_cache(),
  fetch_cache = NULL,
  parallelism = "loop",
  jobs = 1L,
  jobs_preprocess = 1L,
  packages = rev(.packages()),
  lib_loc = NULL,
  prework = character(0),
  prepend = NULL,
  command = NULL,
  args = NULL,
  recipe_command = NULL,
  log_progress = TRUE,
  skip_targets = FALSE,
  timeout = NULL,
  cpu = Inf,
  elapsed = Inf,
  retries = 0,
  force = FALSE,
  graph = NULL,
  trigger = drake::trigger(),
  skip_imports = FALSE,
  skip_safety_checks = FALSE,
  config = NULL,
  lazy_load = "eager",
  session_info = TRUE,
  cache_log_file = NULL,
  seed = NULL,
  caching = "master",
  keep_going = FALSE,
  session = NULL,
  pruning_strategy = NULL,
  makefile_path = NULL,
  console_log_file = NULL,
  ensure_workers = NULL,
  garbage_collection = FALSE,
  template = list(),
  sleep = function(i) 0.01,
  hasty_build = NULL,
  memory_strategy = "speed",
  layout = NULL,
  lock_envir = TRUE,
  history = TRUE,
  recover = FALSE,
  recoverable = TRUE,
  curl_handles = list()
) {
  check_make_call(match.call())
  force(envir)
  if (is.null(config)) {
    config <- drake_config(
      plan = plan,
      targets = targets,
      envir = envir,
      seed = seed,
      verbose = verbose,
      hook = hook,
      parallelism = parallelism,
      jobs = jobs,
      jobs_preprocess = jobs_preprocess,
      packages = packages,
      lib_loc = lib_loc,
      prework = prework,
      prepend = prepend,
      command = command,
      args = args,
      recipe_command = recipe_command,
      log_progress = log_progress,
      cache = cache,
      fetch_cache = fetch_cache,
      timeout = timeout,
      cpu = cpu,
      elapsed = elapsed,
      retries = retries,
      force = force,
      graph = graph,
      trigger = trigger,
      skip_targets = skip_targets,
      skip_imports = skip_imports,
      skip_safety_checks = skip_safety_checks,
      lazy_load = lazy_load,
      session_info = session_info,
      cache_log_file = cache_log_file,
      caching = caching,
      keep_going = keep_going,
      session = session,
      pruning_strategy = pruning_strategy,
      makefile_path = makefile_path,
      console_log_file = console_log_file,
      ensure_workers = ensure_workers,
      garbage_collection = garbage_collection,
      template = template,
      sleep = sleep,
      hasty_build = hasty_build,
      memory_strategy = memory_strategy,
      layout = layout,
      lock_envir = lock_envir,
      history = history,
      recover = recover,
      recoverable = recoverable,
      curl_handles = curl_handles
    )
  }
  config$logger$minor("begin make()")
  runtime_checks(config = config)
  config$running_make <- TRUE
  config$cache$reset_memo_hash()
  on.exit(config$cache$reset_memo_hash())
  config$cache$set(key = "seed", value = config$seed, namespace = "session")
  config$eval[[drake_envir_marker]] <- TRUE
  if (config$log_progress) {
    config$cache$clear(namespace = "progress")
  }
  drake_set_session_info(cache = config$cache, full = config$session_info)
  do_prework(config = config, verbose_packages = config$logger$verbose)
  if (!config$skip_imports) {
    process_imports(config)
  }
  if (is.character(config$parallelism)) {
    config$graph <- outdated_subgraph(config)
  }
  r_make_message(force = FALSE)
  if (!config$skip_targets) {
    process_targets(config)
  }
  drake_cache_log_file_(
    file = config$cache_log_file,
    cache = config$cache,
    jobs = config$jobs_preprocess
  )
  remove(list = names(config$eval), envir = config$eval)
  config$cache$flush_cache()
  if (config$garbage_collection) {
    gc()
  }
  config$logger$minor("end make()")
  invisible()
}

process_targets <- function(config) {
  if (is.character(config$parallelism)) {
    run_native_backend(config)
  } else {
    run_external_backend(config)
  }
}

run_native_backend <- function(config) {
  parallelism <- match.arg(
    config$parallelism,
    c("loop", "clustermq", "future")
  )
  if (igraph::gorder(config$graph)) {
    get(
      paste0("backend_", parallelism),
      envir = getNamespace("drake")
    )(config)
  } else {
    config$logger$major("All targets are already up to date.")
  }
}

run_external_backend <- function(config) {
  warning(
    "`drake` can indeed accept a custom scheduler function for the ",
    "`parallelism` argument of `make()` ",
    "but this is only for the sake of experimentation ",
    "and graceful deprecation. ",
    "Your own custom schedulers may cause surprising errors. ",
    "Use at your own risk.",
    call. = FALSE
  )
  config$parallelism(config = config)
}

outdated_subgraph <- function(config) {
  outdated <- outdated(config, do_prework = FALSE, make_imports = FALSE)
  config$logger$minor("isolate oudated targets")
  igraph::induced_subgraph(graph = config$graph, vids = outdated)
}

drake_set_session_info <- function(
  path = NULL,
  search = NULL,
  cache = drake::drake_cache(path = path),
  verbose = NULL,
  full = TRUE
) {
  deprecate_verbose(verbose)
  if (is.null(cache)) {
    stop("No drake::make() session detected.")
  }
  if (full) {
    cache$set(
      key = "sessionInfo",
      value = utils::sessionInfo(),
      namespace = "session"
    )
  }
  cache$set(
    key = "drake_version",
    value = as.character(utils::packageVersion("drake")),
    namespace = "session"
  )
  invisible()
}

#' @title Do the prework in the `prework`
#'   argument to [make()].
#' \lifecycle{stable}
#' @export
#' @keywords internal
#' @description For internal use only.
#' The only reason this function is exported
#' is to set up parallel socket (PSOCK) clusters
#' without too much fuss.
#' @return Inivisibly returns `NULL`.
#' @param config internal configuration list
#' @param verbose_packages logical, whether to print
#'   package startup messages
#' @examples
#' \dontrun{
#' isolate_example("Quarantine side effects.", {
#' if (suppressWarnings(require("knitr"))) {
#' load_mtcars_example() # Get the code with drake_example("mtcars").
#' # Create a master internal configuration list with prework.
#' con <- drake_config(my_plan, prework = c("library(knitr)", "x <- 1"))
#' # Do the prework. Usually done at the beginning of `make()`,
#' # and for distributed computing backends like "future_lapply",
#' # right before each target is built.
#' do_prework(config = con, verbose_packages = TRUE)
#' # The `eval` element is the environment where the prework
#' # and the commands in your workflow plan data frame are executed.
#' identical(con$eval$x, 1) # Should be TRUE.
#' }
#' })
#' }
do_prework <- function(config, verbose_packages) {
  for (package in union(c("methods", "drake"), config$packages)) {
    expr <- as.call(c(
      quote(require),
      package = package,
      lib.loc = as.call(c(quote(c), config$lib_loc)),
      quietly = TRUE,
      character.only = TRUE
    ))
    if (verbose_packages) {
      expr <- as.call(c(quote(suppressPackageStartupMessages), expr))
    }
    eval(expr, envir = config$eval)
  }
  if (is.character(config$prework)) {
    config$prework <- parse(text = config$prework)
  }
  if (is.language(config$prework)) {
    eval(config$prework, envir = config$eval)
  } else if (is.list(config$prework)) {
    lapply(config$prework, eval, envir = config$eval)
  }
  invisible()
}

# Generate a flat csv log file to represent the state of the cache.
drake_cache_log_file_ <- function(
  file = "drake_cache.csv",
  path = NULL,
  search = NULL,
  cache = drake::drake_cache(path = path),
  verbose = 1L,
  jobs = 1L,
  targets_only = FALSE
) {
  deprecate_search(search)
  if (!length(file) || identical(file, FALSE)) {
    return(invisible())
  } else if (identical(file, TRUE)) {
    file <- formals(drake_cache_log_file_)$file
  }
  out <- drake_cache_log(
    path = path,
    cache = cache,
    verbose = verbose,
    jobs = jobs,
    targets_only = targets_only
  )
  out <- as.data.frame(out)
  # Suppress partial arg match warnings.
  suppressWarnings(
    write.table(
      x = out,
      file = file,
      quote = FALSE,
      row.names = FALSE,
      sep = ","
    )
  )
}

check_make_call <- function(call) {
  x <- names(call)
  if ("config" %in% names(call) && sum(nzchar(x)) > 1L) {
    warning(
      "if you supply a ", shQuote("config"),
      " argument to ", shQuote("make()"),
      " then all additional arguments are ignored. ",
      "For example, in ", shQuote("make(config = config, verbose = 0L)"),
      "verbosity remains at ", shQuote("config$logger$verbose"), ".",
      call. = FALSE
    )
  }
}

runtime_checks <- function(config) {
  assert_config_not_plan(config)
  if (identical(config$skip_safety_checks, TRUE)) {
    return(invisible())
  }
  missing_input_files(config = config)
  subdirectory_warning(config = config)
  assert_outside_cache(config = config)
}

missing_input_files <- function(config) {
  files <- parallel_filter(
    all_imports(config),
    f = is_encoded_path,
    jobs = config$jobs_preprocess
  )
  files <- config$cache$decode_path(x = files)
  missing_files <- files[!file_dep_exists(files)]
  if (length(missing_files)) {
    warning(
      "missing input files:\n",
      multiline_message(missing_files),
      call. = FALSE
    )
  }
  invisible()
}

file_dep_exists <- function(x) {
  file.exists(x) | is_url(x)
}

subdirectory_warning <- function(config) {
  if (identical(Sys.getenv("drake_warn_subdir"), "false")) {
    return()
  }
  dir <- dirname(config$cache$path)
  wd <- getwd()
  if (!length(dir) || wd == dir || is.na(pmatch(dir, wd))) {
    return()
  }
  warning(
    "Running make() in a subdirectory of your project. \n",
    "This could cause problems if your ",
    "file_in()/file_out()/knitr_in() files ",
    "are relative paths.\n",
    "Please either\n",
    "  (1) run make() from your drake project root, or\n",
    "  (2) create a cache in your working ",
    "directory with new_cache('path_name'), or\n",
    "  (3) supply a cache of your own (e.g. make(cache = your_cache))\n",
    "      whose folder name is not '.drake'.\n",
    "  running make() from: ", wd, "\n",
    "  drake project root:  ", dir, "\n",
    "  cache directory:     ", config$cache$path,
    call. = FALSE
  )
}

assert_outside_cache <- function(config) {
  work_dir <- normalizePath(getwd(), mustWork = FALSE)
  cache_dir <- normalizePath(config$cache$path, mustWork = FALSE)
  if (identical(work_dir, cache_dir)) {
    stop(
      "cannot run make() from inside the cache: ", shQuote(cache_dir),
      ". The cache path must be different from your working directory. ",
      "If your drake project lives at ", shQuote("/your/project/root/"), # nolint
      " then you should run ", shQuote("make()"), " from this directory, ",
      "and your cache should be in a subfolder, e.g. ",
      shQuote("/your/project/root/.drake/") # nolint
    )
  }
}

r_make_message <- function(force) {
  r_make_message <- .pkg_envir[["r_make_message"]] %||% TRUE
  on.exit(
    assign(
      x = "r_make_message",
      value = FALSE,
      envir = .pkg_envir,
      inherits = FALSE
    )
  )
  if (force || (r_make_message && sample.int(n = 10, size = 1) < 1.5)) {
    message(
      "In drake, consider r_make() instead of make(). ",
      "r_make() runs make() in a fresh R session ",
      "for enhanced robustness and reproducibility."
    )
  }
}
