#' Graphical goodness-of-fit test
#' 
#' Comparison of the cumulative hazards functions for a semi-parametric and a
#' parametric model.
#' 
#' For the moment only a graphical comparison. The arguments \code{sp} and
#' \code{pp} may be swapped.
#' 
#' @param sp An object of type "coxreg", typically output from
#' \code{\link{coxreg}}
#' @param pp An object of type "phreg", typically output from
#' \code{\link{phreg}}
#' @param main Header for the plot. Default is distribution and "cumulative
#' hazard function"
#' @param col Line colors. should be \code{NULL} (black lines) or of length 2
#' @param printLegend Should a legend be printed? Default is \code{TRUE}.
#' @return No return value.
#' @author Göran Broström
#' @seealso \code{\link{coxreg}} and \code{\link{phreg}}.
#' @keywords distribution
#' @examples
#' 
#' data(mort)
#' oldpar <- par(mfrow = c(2, 2))
#' fit.cr <- coxreg(Surv(enter, exit, event) ~ ses, data = mort)
#' fit.w <- phreg(Surv(enter, exit, event) ~ ses, data = mort)
#' fit.g <- phreg(Surv(enter, exit, event) ~ ses, data = mort,
#' dist = "gompertz")
#' fit.pch <- phreg(Surv(enter, exit, event) ~ ses, data = mort,
#' dist = "pch", cuts = c(5, 10, 15))
#' fit.ev <- phreg(Surv(enter, exit, event) ~ ses, data = mort,
#' dist = "ev")
#' check.dist(fit.cr, fit.w)
#' check.dist(fit.cr, fit.g)
#' check.dist(fit.cr, fit.pch)
#' check.dist(fit.cr, fit.ev)
#' par(oldpar)
#' 
#' @export
check.dist <- function(sp, pp, main = NULL, col = NULL, printLegend = TRUE){
    if (!inherits(sp, "coxreg")){
        if (inherits(pp, "coxreg")){ # swap:
            tmp <- pp
            pp <- sp
            sp <- tmp
            rm(tmp)
        }else{
            stop ("Some argument must be of type 'coxreg'")
        }
    }
    if (!inherits(pp, "phreg"))
        stop ("Some argument must be of type 'phreg' or 'pchreg'")

    if (!sp$nullModel){
        if ((!sp$center) && pp$center)
            warning("The non-parametric fit is not centered.") 
        if ((!pp$center) && sp$center)
            warning("The parametric fit is not centered.")
    }
    if ((!is.null(sp$strata)) || (!is.null(pp$strata)))
        stop("Not for stratified fits; try a comparison stratum by stratum.") 
    if (is.null(main)){
        main <- pp$dist # Capitalize:
        substr(main, 1, 1) <- toupper(substr(main, 1, 1))
        if (main == "Pch") main <- "Piecewise constant"
        if (main == "Ev") main = "Extreme value"
    }

    x.max <- max(pp$y[, 2])
    x <- plot.coxreg(sp, fn = "cum", fig = FALSE)
    if (is.null(x)){
        cat("Error: Must be fixed in check.dist!")
        return(x)
    }
    
    x[[1]][, 2] <- cumsum(x[[1]][, 2]) # Added in 2.4-2
    y.max <- max(x[[1]][, 2])
    if (length(x) > 1){
        for (i in 2:length(x)) y.max <- max(y.max, x[[i]][, 2])
    }
    if (is.null(col)){
        col <- c(1, 1)
    }else{
        if (length(col) != 2) stop("Length of 'col' must be 0 or 2.")
    }
    plot(pp, fn = "cum", fig = TRUE, ## Removed 2.4-0: new.data = pp$means,
         ylim = c(0, y.max), main = main, col = col[1])
    for (rr in 1:length(x)){
        xx <- x[[rr]]
        xx <- rbind(xx, c(x.max, xx[NROW(xx), 2])) # Added 2011-08-10 (2.0-3)
        lines(xx[, 1], xx[, 2], type = "s", lty = 2, col = col[2])
    }
    if (printLegend){
        legend(x = "topleft", legend = c("Parametric", "Non-parametric"),
               lty = 1:2, col = col)
    }
}

