% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_elastic_shape_mean.R
\name{compute_elastic_shape_mean}
\alias{compute_elastic_shape_mean}
\title{Compute an elastic full Procrustes mean for a collection of curves}
\usage{
compute_elastic_shape_mean(
  data_curves,
  knots = seq(0, 1, len = 13),
  type = c("smooth", "polygon"),
  penalty = 2,
  var_type = c("smooth", "constant", "zero"),
  pfit_method = c("smooth", "polygon"),
  smooth_warp = function(i) 0,
  eps = 0.05,
  max_iter = 50,
  verbose = FALSE,
  cluster = NULL
)
}
\arguments{
\item{data_curves}{list of \code{data.frame}s with observed points in each row. Each
variable is one coordinate direction. If there is a variable \code{t},
it is treated as the time parametrization, not as an additional coordinate.}

\item{knots}{set of knots for the mean spline curve}

\item{type}{if "smooth" linear srv-splines are used which results in a differentiable mean curve
if "polygon" the mean will be piecewise linear.}

\item{penalty}{the penalty to use in the covariance smoothing step. use '-1' for no penalty.}

\item{var_type}{(experimental) assume "smooth", "constant" or "zero" measurement-error variance along t}

\item{pfit_method}{(experimental) "smooth" or "polygon"}

\item{smooth_warp}{(experimental) controls the weighting of original and smoothed observations
over the iterations, if pfit_method == "smooth".}

\item{eps}{the algorithm stops if L2 norm of coefficients changes by less than \code{eps}}

\item{max_iter}{maximal number of iterations}

\item{verbose}{print iterations}

\item{cluster}{(experimental) use the parallel package for faster computation}
}
\value{
an object of class \code{elastic_shape_mean}, which is a \code{list}
with entries
\item{type}{"smooth" if mean was modeled using linear srv-splines, "polygon" if constant srv-splines}
\item{coefs}{spline coefficients}
\item{knots}{spline knots}
\item{variance}{sample elastic shape variance}
\item{data_curves}{list of \code{data.frame}s with observed points in each row.
First variable \code{t} gives the initial parametrization, second variable \code{t_optim}
the optimal parametrization when the curve is aligned to the mean. Has the
attributes 'rotation', 'scaling', 'translation' and 'dist_to_mean'. Use
\code{\link{get_procrustes_fit}} to get the elastic full Procrustes fit.}
\item{fit}{see \code{fit_mean}}
}
\description{
Computes an elastic full Procrustes mean for curves stored in \code{data_curves}.
Constructor function for class \code{elastic_shape_mean}.
}
\examples{
curve <- function(t){
  rbind(t*cos(13*t), t*sin(13*t))
}
set.seed(18)
data_curves <- lapply(1:4, function(i){
  m <- sample(10:15, 1)
  delta <- abs(rnorm(m, mean = 1, sd = 0.05))
  t <- cumsum(delta)/sum(delta)
  data.frame(t(curve(t)) + 0.07*t*matrix(cumsum(rnorm(2*length(delta))),
             ncol = 2))
})

#randomly rotate and scale curves
rand_scale <- function(curve){ ( 0.5 + runif(1) ) * curve }
rand_rotate <- function(curve){
  names <- colnames(curve)
  theta <- 2*pi*runif(1)
  mat <- matrix(c(cos(theta), sin(theta), -sin(theta), cos(theta)), nrow = 2, ncol = 2)
  curve.rot <- as.matrix(curve) \%*\% t(mat)
  curve.rot <- as.data.frame(curve.rot)
  colnames(curve.rot) <- names
  return(curve.rot)
}
data_curves <- lapply(data_curves, rand_scale)
data_curves <- lapply(data_curves, rand_rotate)

#compute smooth procrustes mean with 2 order penalty
knots <- seq(0,1, length = 11)
elastic_shape_mean <- compute_elastic_shape_mean(
    data_curves,
    knots = knots,
    type = "smooth",
    penalty = 2
    )
plot(elastic_shape_mean)
}
