% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/job.R
\docType{class}
\name{EplusJob}
\alias{EplusJob}
\title{Run EnergyPlus Simulation and Collect Outputs}
\description{
\code{EplusJob} class wraps the EnergyPlus command line interface and provides
methods to extract simulation outputs.
}
\details{
eplusr uses the EnergyPlus SQL output for extracting simulation outputs.
\code{EplusJob} has provide some wrappers that do SQL query to get report data
results, i.e. results from \code{Output:Variable} and \code{Output:Meter*}. But for
\code{Output:Table} results, you have to be familiar with the structure of the
EnergyPlus SQL results, especially for table \emph{"TabularDataWithStrings"}. For
details, please see \emph{"2.20 eplusout.sql"}, especially \emph{"2.20.4.4 TabularData
Table"} in EnergyPlus \emph{"Output Details and Examples"} documentation.
}
\section{NOTE}{


When using \code{$run()} in \link{Idf} class, which internally creates an
\code{EplusJob} object and calls its \code{$run()} method, an object in
\code{Output:SQLite} with \code{Option Type} value of \code{SimpleAndTabular} will be
automatically created if it does not exists.

However, when creating an \code{EplusJob} using \code{\link[=eplus_job]{eplus_job()}}, the IDF file is not
parsed but directly pass its path to EnergyPlus. Thus, that process of
handling \code{Output:SQLite} class is not performed. If you want to ensure
that the output collection functionality in \code{EplusJob} class works
successfully, it is recommended to first read that IDF file using
\code{\link[=read_idf]{read_idf()}} and then use \code{$run()} method in \link{Idf} class by doing
\code{idf$run()}.
}

\section{Usage}{
\preformatted{job$run(wait = TRUE)
job$kill()
job$status()
job$errors(info = FALSE)
job$output_dir(open = FALSE)
job$locate_output(suffix = ".err", strict = TRUE)
job$report_data_dict()
job$report_data(key_value = NULL, name = NULL, year = NULL, tz = "GMT", case = "auto")
job$tabular_data()
job$clone(deep = FALSE)
job$print()
}
}

\section{Create}{
\preformatted{job <- eplus_job(idf, epw)
}

\strong{Arguments}
\itemize{
\item \code{idf}: Path to an local EnergyPlus IDF file or an \code{Idf} object.
\item \code{epw}: Path to an local EnergyPlus EPW file or an \code{Epw} object.
}
}

\section{Basic info}{
\preformatted{job$path(type = c("all", "idf", "epw"))
}

\code{$path()} returns the path of IDF or EPW of current job.

\strong{Arguments}
\itemize{
\item \code{type}: If \code{"all"}, both the IDF path and EPW path are returned. If
\code{"idf"}, only IDF path is returned. If \code{"epw"}, only EPW path is
returned.  Default: \code{"all"}.
}
}

\section{Run}{
\preformatted{job$run(wait = TRUE)
job$kill()
job$status()
}

\code{$run()} runs the simulation using input model and weather file.
If \code{wait} is FALSE, then the job will be run in the background. You can
get updated job status by just print the EplusJob object.

\code{$kill()} kills the background EnergyPlus process if possible. It only
works when simulation runs in non-waiting mode.

\code{$status()} returns a named list of values indicates the status of the job:
\itemize{
\item \code{run_before}: \code{TRUE} if the job has been run before. \code{FALSE} otherwise.
\item \code{alive}: \code{TRUE} if the simulation is still running in the background.
\code{FALSE} otherwise.
\item \code{terminated}: \code{TRUE} if the simulation was terminated during last
simulation. \code{FALSE} otherwise. \code{NA} if the job has not been run yet.
\item \code{successful}: \code{TRUE} if last simulation ended successfully. \code{FALSE}
otherwise. \code{NA} if the job has not been run yet.
\item \code{changed_after}: \code{TRUE} if the IDF file has been changed since last
simulation. \code{FALSE} otherwise. \code{NA} if the job has not been run yet.
}

\strong{Arguments}
\itemize{
\item \code{wait}: If \code{TRUE}, R will hang on and wait for the simulation to complete.
EnergyPlus standard output (stdout) and error (stderr) is printed to the
R console. If \code{FALSE}, simulation will be run in a background process.
Default: \code{TRUE}.
}
}

\section{Results Extraction}{
\preformatted{job$output_dir(open = FALSE)
job$locate_output(suffix = ".err", strict = TRUE)
job$report_data_dict()
job$report_data(key_value = NULL, name = NULL, year = NULL, tz = "GMT", case = "auto")
job$tabular_data()
}

\code{$output_dir()} returns the output directory of simulation results.

\code{$locate_output()} returns the path of a single output file specified by file
suffix.

\code{$report_data_dict()} returns a data.table which contains all information about
report data. For details on the meaning of each columns, please see
"2.20.2.1 ReportDataDictionary Table" in EnergyPlus "Output Details and
Examples" documentation.

\code{$report_data()} extracts the report data in a data.table using key values
and variable names.

\code{$tabular_data()} extracts all tabular data in a data.table.

\strong{Arguments}:
\itemize{
\item \code{open}: If \code{TRUE}, the output directory will be opened. It may only work
well on Windows.
\item \code{suffix}: A string that indicates the file suffix of simulation output.
Default: \code{".err"}.
\item \code{strict}: If \code{TRUE}, it will check if the simulation was terminated, is
still running or the file exists or not. Default: \code{TRUE}.
\item \code{key_value}: A character vector to identify key name of the data. If
\code{NULL}, all keys of that variable will be returned. Default: \code{NULL}.
\item \code{name}: A character vector to specify the actual data name. If \code{NULL}, all
variables will be returned. Default: \code{NULL}.
\item \code{year}: The year of the date and time in column \code{DateTime}. If \code{NULL}, it
will be the current year. Default: \code{NULL}
\item \code{tz}: Time zone of date and time in column \code{DateTime}. Default: \code{"GMT"}.
\item \code{case}: If not \code{NULL}, a character column will be added indicates the case
of this simulation. If \code{"auto"}, the name of the IDF file will be used.
}
}

\section{Clone}{
\preformatted{job$clone(deep = FALSE)
}

\code{$clone()} copies and returns the cloned job. Because \code{EplusJob} uses
\code{R6Class} under the hook which has "modify-in-place" semantics, \code{job_2 <- job_1} does not copy \code{job_1} at all but only create a new binding to
\code{job_1}.  Modify \code{job_1} will also affect \code{job_2} as well, as these two
are exactly the same thing underneath. In order to create a complete
cloned copy, please use \code{$clone(deep = TRUE)}.

\strong{Arguments}
\itemize{
\item \code{deep}: Has to be \code{TRUE} if a complete cloned copy is desired.
}
}

\section{Printing}{
\preformatted{job$print()
print(job)
}

\code{$print()} shows the core information of this EplusJob, including the
path of model and weather, the version and path of EnergyPlus used
to run simulations, and the simulation job status.

\code{$print()} is quite useful to get the simulation status, especially when
\code{wait} is \code{FALSE} in \code{$run()}. The job status will be updated and printed
whenever \code{$print()} is called.
}

\examples{
\dontrun{
if (is_avail_eplus(8.8)) {
    idf_name <- "1ZoneUncontrolled.idf"
    epw_name <-  "USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"

    idf_path <- file.path(eplus_config(8.8)$dir, "ExampleFiles", idf_name)
    epw_path <- file.path(eplus_config(8.8)$dir, "WeatherData", epw_name)

    # copy to tempdir
    file.copy(c(idf_path, epw_path), tempdir())

    # create an EplusJob from local an IDF and an EPW file
    job <- eplus_job(file.path(tempdir(), idf_name), file.path(tempdir(), epw_name))

    # get paths of idf and epw
    job$path("all")
    job$path("idf")
    job$path("epw")

    # get current job status
    job$status()

    # check if the job has been run before
    job$status()$run_before

    # run the job in waiting mode
    job$run(wait = TRUE)

    # check the job status again
    job$status()$run_before
    job$status()$successful

    # get output directory
    job$output_dir()

    # open the output directory
    job$output_dir(open = TRUE)

    # check simulation errors
    job$errors()

    # check simulation errors, only including warnings and errors
    job$errors(info = FALSE)

    # get the file path of an output with a given suffix
    job$locate_output(".err")

    # give an error when simulation did not complete successfully or that file
    # does not exist
    job$locate_output(".exe", strict = TRUE)

    # retrieve simulation results will fail if there is no EnergyPlus SQL output.
    job$report_data_dict()

    # instead, using `$run()` method in Idf class, which will add an
    # `Output:SQLite` object automatically
    idf <- read_idf(file.path(tempdir(), idf_name))
    job <- idf$run(file.path(tempdir(), epw_name), dir = NULL)

    # get report data dictionary
    str(job$report_data_dict())

    # extract all report data
    str(job$report_data())

    # extract some report variable
    str(job$report_data(name = "EnergyTransfer:Building", case = NULL))

    # add a "Case" column in the returned data.table
    str(job$report_data(name = "EnergyTransfer:Building", case = "Test"))

    # change the format of datetime column in the returned data.table
    str(job$report_data(name = "EnergyTransfer:Building", year = 2016L, tz = Sys.timezone()))

    # get all tabular data
    str(job$tabular_data())
}
}
}
\seealso{
\link{ParametricJob} class for EnergyPlus parametric simulations.
}
\author{
Hongyuan Jia
}
