% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_turbulence.R
\name{model_turbulence}
\alias{model_turbulence}
\title{Model the seismic spectrum due to hydraulic turbulence}
\usage{
model_turbulence(
  d_s,
  s_s,
  r_s = 2650,
  h_w,
  w_w,
  a_w,
  f = c(1, 100),
  r_0,
  f_0,
  q_0,
  v_0,
  p_0,
  n_0,
  res = 1000,
  eseis = FALSE,
  ...
)
}
\arguments{
\item{d_s}{\code{Numeric} value, mean sediment grain diameter (m)}

\item{s_s}{\code{Numeric} value, standard deviation of sediment grain 
diameter (m)}

\item{r_s}{\code{Numeric} value, specific sediment density (kg / m^3)}

\item{h_w}{\code{Numeric} value, fluid flow depth (m)}

\item{w_w}{\code{Numeric} value, fluid flow width (m)}

\item{a_w}{\code{Numeric} value, fluid flow inclination angle (radians)}

\item{f}{\code{Numeric} vector, frequency range to be modelled. 
If of length two the argument is interpreted as representing the lower and 
upper limit and the final length of the frequency vector is set by the 
argument \code{res}. If \code{f} contains more than two values it is 
interpreted as the actual frequency vector and the value of \code{res} is 
ignored.}

\item{r_0}{\code{Numeric} value, distance of seismic station to source}

\item{f_0}{\code{Numeric} value, reference frequency (Hz)}

\item{q_0}{\code{Numeric} value, ground quality factor at \code{f_0}}

\item{v_0}{\code{Numeric} value, phase speed of the Rayleigh wave at 
\code{f_0} (m/s)}

\item{p_0}{\code{Numeric} value, variation exponent of Rayleigh wave 
velocities with frequency (dimensionless)}

\item{n_0}{\code{Numeric} vector of length two, Greens function 
displacement amplitude coefficients. Cf. N_ij in eq. 36 in Gimbert et 
al. (2014)}

\item{res}{\code{Numeric} value, output resolution, i.e. length of the 
spectrum vector. Default is 1000.}

\item{eseis}{\code{Character} value, option to return an eseis object 
instead of a data frame. Default is \code{FALSE}.}

\item{\dots}{Further arguments passed to the function.}
}
\value{
\code{eseis} object containing the modelled spectrum.
}
\description{
The function calculates the seismic spectrum as predicted by the model 
of Gimbert et al. (2014) for hydraulic turbulence. The code was written to 
R by Sophie Lagarde and integrated to the R package 'eseis' by Michael 
Dietze.
}
\details{
The model uses a set of predefined constants. These can also be changed
by the user, using the \code{...} argument:
\itemize{
  \item \code{c = 0.5}, instantaneous fluid-grain friction coefficient 
  (dimensionless)
  \item \code{g = 9.81}, gravitational acceleration (m/s^2)
  \item \code{k = 0.5}, Kolmogrov constant (dimensionless)
  \item \code{k_s = 3 * d_s}, roughness length (m)
  \item \code{h = k_s / 2}, reference height of the measurement (m)
  \item \code{e_0 = 0}, exponent of Q increase with frequency 
  (dimensionless)
  \item \code{r_w = 1000}, specific density of the fluid (kg/m^3)
  \item \code{c_w = 0.5}, instantaneous fluid-grain friction coefficient
  (dimensionless)
}
}
\examples{

## model the turbulence-related power spectrum
P <- model_turbulence(d_s = 0.03, # 3 cm mean grain-size
                      s_s = 1.35, # 1.35 log standard deviation
                      r_s = 2650, # 2.65 g/cm^3 sediment density
                      h_w = 0.8, # 80 cm water level
                      w_w = 40, # 40 m river width
                      a_w = 0.0075, # 0.0075 rad river inclination
                      f = c(1, 200), # 1-200 Hz frequency range
                      r_0 = 10, # 10 m distance to the river
                      f_0 = 1, # 1 Hz Null frequency 
                      q_0 = 10, # 10 quality factor at f = 1 Hz
                      v_0 = 2175, # 2175 m/s phase velocity
                      p_0 = 0.48, # 0.48 power law variation coefficient
                      n_0 = c(0.6, 0.8), # Greens function estimates
                      res = 1000) # 1000 values build the output resolution

## plot the power spectrum
plot_spectrum(data = P)
              
}
\author{
Sophie Lagarde, Michael Dietze
}
\keyword{eseis}
