% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/euler.R
\name{euler}
\alias{euler}
\alias{euler.default}
\alias{euler.data.frame}
\alias{euler.matrix}
\alias{euler.table}
\alias{euler.list}
\title{Area-proportional Euler diagrams}
\usage{
euler(combinations, ...)

\method{euler}{default}(combinations, input = c("disjoint", "union"),
  shape = c("circle", "ellipse"), control = list(), ...)

\method{euler}{data.frame}(combinations, weights = NULL, by = NULL,
  sep = "_", factor_names = TRUE, ...)

\method{euler}{matrix}(combinations, ...)

\method{euler}{table}(combinations, ...)

\method{euler}{list}(combinations, ...)
}
\arguments{
\item{combinations}{set relationships as a named numeric vector, matrix, or
data.frame (see \strong{methods (by class)})}

\item{...}{arguments passed down to other methods}

\item{input}{type of input: disjoint identities
(\code{'disjoint'}) or unions (\code{'union'}).}

\item{shape}{geometric shape used in the diagram}

\item{control}{a list of control parameters.
\itemize{
\item \code{extraopt}: should the more thorough optimizer (currently
\code{\link[GenSA:GenSA]{GenSA::GenSA()}}) kick in (provided \code{extraopt_threshold} is exceeded)? The
default is \code{TRUE} for ellipses and three sets and \code{FALSE} otherwise.
\item \code{extraopt_threshold}: threshold, in terms of \code{diagError}, for when
the extra optimizer kicks in. This will almost always slow down the
process considerably. A value of 0 means
that the extra optimizer will kick in if there is \emph{any} error. A value of
1 means that it will never kick in. The default is \code{0.001}.
\item \code{extraopt_control}: a list of control parameters to pass to the
extra optimizer, such as \code{max.call}. See \code{\link[GenSA:GenSA]{GenSA::GenSA()}}.
}}

\item{weights}{a numeric vector of weights of the same length as
the number of rows in \code{combinations}.}

\item{by}{a factor or character matrix to be used in \code{\link[base:by]{base::by()}} to
split the data.frame or matrix of set combinations}

\item{sep}{a character to use to separate the dummy-coded factors
if there are factor or character vectors in 'combinations'.}

\item{factor_names}{whether to include factor names when
constructing dummy codes}
}
\value{
A list object of class \code{'euler'} with the following parameters.
\item{ellipses}{a matrix of \code{h} and \code{k} (x and y-coordinates for the
centers of the shapes), semiaxes \code{a} and \code{b}, and rotation angle \code{phi}}
\item{original.values}{set relationships in the input}
\item{fitted.values}{set relationships in the solution}
\item{residuals}{residuals}
\item{regionError}{the difference in percentage points between each
disjoint subset in the input and the respective area in the output}
\item{diagError}{the largest \code{regionError}}
\item{stress}{normalized residual sums of squares}
}
\description{
Fit Euler diagrams (a generalization of Venn diagrams) using numerical
optimization to find exact or approximate solutions to a specification of set
relationships. The shape of the diagram may be a circle or an ellipse.
}
\details{
If the input is a matrix or data frame and argument \code{by} is specified,
the function returns a list of euler diagrams.

The function minimizes the residual sums of squares,
\deqn{
  \sum_{i=1}^n (A_i - \omega_i)^2,
}{
  \sum (A_i - \omega_i)^2,
}
where \eqn{\omega_i} the size of the ith disjoint subset, and \eqn{A_i} the
corresponding area in the diagram, that is, the unique contribution to the
total area from this overlap.

\code{\link[=euler]{euler()}} also returns \code{stress} (from \pkg{venneuler}), as well as
\code{diagError}, and \code{regionError} from \pkg{eulerAPE}.

The \emph{stress} statistic is computed as

\deqn{
  \frac{\sum_{i=1}^n (A_i - \beta\omega_i)^2}{\sum_{i=1}^n A_i^2},
}{
  \sum (A_i - \beta\omega_i)^2 / \sum A_i^2,
}
where
\deqn{
  \beta = \sum_{i=1}^n A_i\omega_i / \sum_{i=1}^n \omega_i^2.
}{
  \beta = \sum A_i\omega_i / \sum \omega_i^2.
}

\code{regionError} is computed as

\deqn{
  \left| \frac{A_i}{\sum_{i=1}^n A_i} - \frac{\omega_i}{\sum_{i=1}^n \omega_i}\right|.
}{
  max|A_i / \sum A  - \omega_i / \sum \omega|.
}

\code{diagError} is simply the maximum of regionError.
}
\section{Methods (by class)}{
\itemize{
\item \code{default}: a named numeric vector, with
combinations separated by an ampersand, for instance \code{A&B = 10}.
Missing combinations are treated as being 0.

\item \code{data.frame}: a \code{data.frame} of logicals, binary integers, or
factors.

\item \code{matrix}: a matrix that can be converted to a data.frame of logicals
(as in the description above) via \code{\link[base:as.data.frame.matrix]{base::as.data.frame.matrix()}}.

\item \code{table}: A table with \code{max(dim(x)) < 3}.

\item \code{list}: a list of vectors, each vector giving the contents of
that set (with no duplicates). Vectors in the list do not need to be named.
}}

\examples{
# Fit a diagram with circles
combo <- c(A = 2, B = 2, C = 2, "A&B" = 1, "A&C" = 1, "B&C" = 1)
fit1 <- euler(combo)

# Investigate the fit
fit1

# Refit using ellipses instead
fit2 <- euler(combo, shape = "ellipse")

# Investigate the fit again (which is now exact)
fit2

# Plot it
plot(fit2)

# A set with no perfect solution
euler(c("a" = 3491, "b" = 3409, "c" = 3503,
        "a&b" = 120, "a&c" = 114, "b&c" = 132,
        "a&b&c" = 50))

# Using grouping via the 'by' argument through the data.frame method
euler(fruits, by = list(sex, age))


# Using the matrix method
euler(organisms)

# Using weights
euler(organisms, weights = c(10, 20, 5, 4, 8, 9, 2))

# The table method
euler(pain, factor_names = FALSE)

# A euler diagram from a list of sample spaces (the list method)
euler(plants[c("erigenia", "solanum", "cynodon")])
}
\references{
Wilkinson L. Exact and Approximate Area-Proportional Circular
Venn and Euler Diagrams. IEEE Transactions on Visualization and Computer
Graphics (Internet). 2012 Feb (cited 2016 Apr 9);18(2):321-31. Available
from:
\url{http://doi.org/10.1109/TVCG.2011.56}

Micallef L, Rodgers P. eulerAPE: Drawing Area-Proportional 3-Venn Diagrams
Using Ellipses. PLOS ONE (Internet). 2014 Jul (cited 2016 Dec 10);9(7):e101717. Available from:
\url{http://dx.doi.org/10.1371/journal.pone.0101717}
}
\seealso{
\code{\link[=plot.euler]{plot.euler()}}, \code{\link[=print.euler]{print.euler()}}, \code{\link[=eulerr_options]{eulerr_options()}}, \code{\link[=venn]{venn()}}
}
