% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelling.R
\name{get_energy_models}
\alias{get_energy_models}
\title{Get a tibble of energy GMM for every user profile}
\usage{
get_energy_models(sessions_profiles, log = TRUE, by_power = FALSE)
}
\arguments{
\item{sessions_profiles}{tibble, sessions data set in evprof
\href{https://mcanigueral.github.io/evprof/articles/sessions-format.html}{standard format}
with user profile attribute \code{Profile}}

\item{log}{logical, whether to transform \code{ConnectionStartDateTime} and
\code{ConnectionHours} variables to natural logarithmic scale (base = \code{exp(1)}).}

\item{by_power}{Logical, true to fit the energy models for every charging rate separately}
}
\value{
tibble
}
\description{
Get a tibble of energy GMM for every user profile
}
\examples{
\donttest{
library(dplyr)

# Select working day sessions (`Timecycle == 1`) that
# disconnect the same day (`Disconnection == 1`)
sessions_day <- california_ev_sessions \%>\%
  divide_by_timecycle(
    months_cycles = list(1:12), # Not differentiation between months
    wdays_cycles = list(1:5, 6:7) # Differentiation between workdays/weekends
  ) \%>\%
  divide_by_disconnection(
    division_hour = 10, start = 3
  ) \%>\%
  filter(
    Disconnection == 1, Timecycle == 1
  )
plot_points(sessions_day, start = 3)

# Identify two clusters
sessions_clusters <- cluster_sessions(
  sessions_day, k=2, seed = 1234, log = TRUE
)

# Plot the clusters found
plot_bivarGMM(
  sessions = sessions_clusters$sessions,
  models = sessions_clusters$models,
  log = TRUE, start = 3
)

# Define the clusters with user profile interpretations
clusters_definitions <- define_clusters(
  models = sessions_clusters$models,
  interpretations = c(
    "Connections during working hours",
    "Connections during all day (high variability)"
  ),
  profile_names = c("Workers", "Visitors"),
  log = TRUE
)

# Classify each session to the corresponding user profile
sessions_profiles <- set_profiles(
  sessions_clustered = list(sessions_clusters$sessions),
  clusters_definition = list(clusters_definitions)
)

# Create a table with the energy GMM parameters
get_energy_models(sessions_profiles, log = TRUE)

# If there is a `Power` variable in the data set
# you can create an energy model per power rate and user profile
# First it is convenient to round the `Power` values for more generic models
sessions_profiles <- sessions_profiles \%>\%
  mutate(Power = round_to_interval(Power, 3.7)) \%>\%
  filter(Power < 11)
sessions_profiles$Power[sessions_profiles$Power == 0] <- 3.7
get_energy_models(sessions_profiles, log = TRUE, by_power = TRUE)
}



}
