
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU Library General 
# Public License along with this library; if not, write to the 
# Free Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
# MA  02111-1307  USA

# Copyright (C) 1998-2003 by Diethelm Wuertz


################################################################################
# FUNCTION:             VARIANCE-1 STUDENT-T Distribution:
#  dstd                   Density for the Student-t Distribution
#  pstd                   Probability function for the Student-t Distribution
#  qstd                   Quantile function for the Student-t Distribution
#  rstd                   Random Number Generator for the Student-t
# FUNCTION:             DESCRIPTION:
#  dsstd                  Density for the skewed Student-t Distribution
#  psstd                  Probability function for the skewed STD
#  qssts                  Quantile function for the skewed STD
#  rsstd                  Random Number Generator for the skewed STD
################################################################################


################################################################################
# STUDENT-T VARIANCE-1 DISTRIBUTION


dstd =
function(x, mean = 0, sd = 1, nu = 5)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Compute the density for the  
    #   Student-t distribution.
    
    # FUNCTION:
    
    # Compute Density:
    s = sqrt(nu/(nu-2))
    z = (x - mean) / sd
    # result = .Internal(dnt(x = z*s, df = nu, ncp = 0, log = FALSE)) / (sd/s)
    result = dt(x = z*s, df = nu) * s / sd
    
    # Return Value:
    result
}


# ------------------------------------------------------------------------------


pstd =
function (q, mean = 0, sd = 1, nu = 5)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Compute the probability for the  
    #   Student-t distribution.
    
    # FUNCTION:
    
    # Compute Probability:
    s = sqrt(nu/(nu-2))
    z = (q - mean) / sd
    # result = .Internal(pnt(q = z*s, df = nu, ncp = 0, lower.tail = TRUE, 
    #   log.p = FALSE))
    result = pt(q = z*s, df = nu)
    
    # Return Value:
    result
}


# ------------------------------------------------------------------------------


qstd =
function (p, mean = 0, sd = 1, nu = 5)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Compute the quantiles for the  
    #   Student-t distribution.
    
    # FUNCTION:
    
    # Compute Quantiles:
    s = sqrt(nu/(nu-2))
    # x = .Internal(qt(p = p, df = nu, lower.tail = TRUE, log.p = FALSE)) / s
    # result = x*sd + mean
    result = qt(p = p, df = nu) * sd / s + mean
    
    # Return Value:
    result
}


# ------------------------------------------------------------------------------


rstd =
function(n, mean = 0, sd = 1, nu = 5)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Generate random deviates from the  
    #   Student-t distribution.
    
    # FUNCTION:
    
    # Generate Random Deviates:
    s = sqrt(nu/(nu-2))
    # result = .Internal(rt(n = n, df = nu)) * sd / s + mean
    result = rt(n = n, df = nu) * sd / s + mean 
    
    # Return Value:
    result
}


################################################################################
# SKEW STUDENT-T VARIANCE-1 DISTRIBUTION


dsstd =
function(x, mean = 0, sd = 1, nu = 5, xi = 1.5)
{	# A function implemented by Diethelm Wuertz 

	# Description:
	# 	Compute the density function of the 
	#	skewed Student-t distribution
	
	# FUNCTION:
	
	# Internal Function:
	.dsstd = function(x, nu, xi) {
		# Standardize:	
		  m1 = 2 * sqrt(nu-2) / (nu-1) / beta(1/2, nu/2)
		  mu = m1*(xi-1/xi)
		  sigma =  sqrt((1-m1^2)*(xi^2+1/xi^2) + 2*m1^2 - 1)
		  z = x*sigma + mu  
		# Compute:
		  Xi = xi^sign(z)
		  g = 2 / (xi + 1/xi)	
		  Density = g * dstd(x = z/Xi, nu = nu)  
		# Return Value:
		  Density * sigma }
	
    # Shift and Scale:
	result = .dsstd(x = (x-mean)/sd, nu = nu, xi = xi) / sd
	
	# Return Value:
	result
}


# ------------------------------------------------------------------------------


psstd =
function(q, mean = 0, sd = 1, nu = 5, xi = 1.5)
{	# A function implemented by Diethelm Wuertz 

	# Description:
	# 	Compute the distribution function of the 
	#	skewed Student-t distribution
	
	# FUNCTION:
	
	# Internal Function:
	.psstd = function(q, nu, xi) {
		# Standardize:
		  m1 = 2 * sqrt(nu-2) / (nu-1) / beta(1/2, nu/2)
		  mu = m1*(xi-1/xi)
		  sigma =  sqrt((1-m1^2)*(xi^2+1/xi^2) + 2*m1^2 - 1)
		  z = q*sigma + mu
		# Compute:	
		  Xi = xi^sign(z)
		  g = 2 / (xi + 1/xi)	
		  Probability = H(z) - sign(z) * g * Xi * pstd(q = -abs(z)/Xi, nu = nu)
		# Return Value:
		  Probability }
		  
	# Shift and Scale:
	result = .psstd(q = (q-mean)/sd, nu = nu, xi = xi)
		  
	# Return Value:
	result
}


# ------------------------------------------------------------------------------	

	
qsstd =
function(p, mean = 0, sd = 1, nu = 5, xi = 1.5)
{	# A function implemented by Diethelm Wuertz 

	# Description:
	# 	Compute the quantile function of the 
	#	skewed Student-t distribution
	
	# FUNCTION:
	
	# Internal Function:
	.qsstd = function(p, nu, xi) {
		# Standardize:
		  m1 = 2 * sqrt(nu-2) / (nu-1) / beta(1/2, nu/2)
		  mu = m1*(xi-1/xi)
		  sigma =  sqrt((1-m1^2)*(xi^2+1/xi^2) + 2*m1^2 - 1)
		# Compute:	
		  g = 2  / (xi + 1/xi)
		  sig = sign(p-1/2) 
		  Xi = xi^sig		  
		  p = (H(p-1/2)-sig*p) / (g*Xi)
		  Quantile = (-sig*qstd(p=p, sd=Xi, nu=nu) - mu ) / sigma
		# Return Value:
		  Quantile }
	
	# Shift and Scale:
	result = .qsstd(p = p, nu = nu, xi = xi) * sd + mean
	
	# Return Value:
	result
}

	
# ------------------------------------------------------------------------------
	

rsstd =
function(n, mean = 0, sd = 1, nu = 5, xi = 1.5)
{	# A function implemented by Diethelm Wuertz 

	# Description:
	# 	Generate random deviates from the 
	#	skewed Student-t distribution
	
	# FUNCTION:
	
	# Internal Function:
	.rsstd = function(n, nu, xi) {
		# Generate Random Deviates:
		  weight = xi / (xi + 1/xi)
		  z = runif(n, -weight, 1-weight)
		  Xi = xi^sign(z)
		  Random = -abs(rstd(n, nu=nu))/Xi * sign(z)	
		# Scale:
		  m1 = 2 * sqrt(nu-2) / (nu-1) / beta(1/2, nu/2)
		  mu = m1*(xi-1/xi)
		  sigma =  sqrt((1-m1^2)*(xi^2+1/xi^2) + 2*m1^2 - 1)
		  Random = (Random - mu ) / sigma	
		# Return value:
		  Random }
	
	# Shift and Scale:
	result = .rsstd(n = n, nu = nu, xi = xi) * sd + mean
	
	# Return Value:
	result
}


# ------------------------------------------------------------------------------

