% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/variable_creation_functions.R
\name{draw_discrete}
\alias{draw_discrete}
\alias{draw_binomial}
\alias{draw_categorical}
\alias{draw_ordered}
\alias{draw_count}
\alias{draw_binary}
\alias{draw_likert}
\alias{draw_quantile}
\title{Draw discrete variables including binary, binomial count, poisson count,
ordered, and categorical}
\usage{
draw_binomial(prob = link(latent), trials = 1, N = length(prob),
  latent = NULL, link = "identity", quantile_y = NULL)

draw_categorical(prob = link(latent), N = NULL, latent = NULL,
  link = "identity", category_labels = NULL)

draw_ordered(x = link(latent), breaks = c(-1, 0, 1),
  break_labels = NULL, N = length(x), latent = NULL,
  strict = FALSE, link = "identity")

draw_count(mean = link(latent), N = length(mean), latent = NULL,
  link = "identity", quantile_y = NULL)

draw_binary(prob = link(latent), N = length(prob), link = "identity",
  latent = NULL, quantile_y = NULL)

draw_likert(x, type = 7, breaks = NULL, N = length(x),
  latent = NULL, link = "identity", strict = !is.null(breaks))

draw_quantile(type, N)
}
\arguments{
\item{prob}{A number or vector of numbers representing the probability for
binary or binomial outcomes; or a number, vector, or matrix of numbers
representing probabilities for categorical outcomes. If you supply a link
function, these underlying probabilities will be transformed.}

\item{trials}{for \code{draw_binomial}, the number of trials for each observation}

\item{N}{number of units to draw. Defaults to the length of the vector of
probabilities or latent data you provided.}

\item{latent}{If the user provides a link argument other than identity, they
should provide the variable \code{latent} rather than \code{prob} or \code{mean}}

\item{link}{link function between the latent variable and the probability of
a positive outcome, e.g. "logit", "probit", or "identity". For the "identity"
link, the latent variable must be a probability.}

\item{quantile_y}{A vector of quantiles; if provided, rather than drawing
stochastically from the distribution of interest, data will be drawn at
exactly those quantiles.}

\item{category_labels}{vector of labels for the categories produced by
\code{draw_categorical}. If provided, must be equal to the number of categories
provided in the \code{prob} argument.}

\item{x}{for \code{draw_ordered} or \code{draw_likert}, the latent data for each
observation.}

\item{breaks}{vector of breaks to cut a latent outcome into ordered
categories with \code{draw_ordered} or \code{draw_likert}}

\item{break_labels}{vector of labels for the breaks to cut a latent outcome
into ordered categories with \code{draw_ordered}. (Optional)}

\item{strict}{Logical indicating whether values outside the provided breaks should be coded as NA. Defaults to \code{FALSE}, in which case effectively additional breaks are added between -Inf and the lowest break and between the highest break and Inf.}

\item{mean}{for \code{draw_count}, the mean number of count units for each observation}

\item{type}{Type of Likert scale data for \code{draw_likert}. Valid options are 4,
5, and 7. Type corresponds to the number of categories in the Likert scale.}
}
\value{
A vector of data in accordance with the specification; generally
numeric but for some functions, including \code{draw_ordered}, may be factor if
break labels are provided.
}
\description{
Drawing discrete data based on probabilities or latent traits is a common
task that can be cumbersome. Each function in our discrete drawing set creates
a different type of discrete data: \code{draw_binary} creates binary 0/1 data,
\code{draw_binomial} creates binomial data (repeated trial binary data),
\code{draw_categorical} creates categorical data, \code{draw_ordered}
transforms latent data into observed ordered categories, \code{draw_count}
creates count data (poisson-distributed). \code{draw_likert} is an alias to
\code{draw_ordered} that pre-specifies break labels and offers default breaks
appropriate for a likert survey question.
}
\details{
For variables with intra-cluster correlations, see
\code{\link{draw_binary_icc}} and \code{\link{draw_normal_icc}}
}
\examples{

# Drawing binary values (success or failure, treatment assignment)
fabricate(N = 3,
   p = c(0, .5, 1),
   binary = draw_binary(prob = p))

# Drawing binary values with probit link (transforming continuous data
# into a probability range).
fabricate(N = 3,
   x = 10 * rnorm(N),
   binary = draw_binary(latent = x, link = "probit"))

# Repeated trials: `draw_binomial`
fabricate(N = 3,
   p = c(0, .5, 1),
   binomial = draw_binomial(prob = p, trials = 10))

# Ordered data: transforming latent data into observed, ordinal data.
# useful for survey responses.
fabricate(N = 3,
   x = 5 * rnorm(N),
   ordered = draw_ordered(x = x,
                          breaks = c(-Inf, -1, 1, Inf)))

# Providing break labels for latent data.
fabricate(N = 3,
   x = 5 * rnorm(N),
   ordered = draw_ordered(x = x,
                          breaks = c(-Inf, -1, 1, Inf),
                          break_labels = c("Not at all concerned",
                                           "Somewhat concerned",
                                           "Very concerned")))

# Likert data: often used for survey data
fabricate(N = 10,
          support_free_college = draw_likert(x = rnorm(N),
                                             type = 5))

# Count data: useful for rates of occurrences over time.
fabricate(N = 5,
   x = c(0, 5, 25, 50, 100),
   theft_rate = draw_count(mean=x))

# Categorical data: useful for demographic data.
fabricate(N = 6, p1 = runif(N), p2 = runif(N), p3 = runif(N),
          cat = draw_categorical(cbind(p1, p2, p3)))

}
