% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fetch.R
\name{fetch}
\alias{fetch}
\title{Calculate Wind Fetch}
\usage{
fetch(polygon_layer, site_layer, max_dist = 300, n_directions = 9,
  site_names, quiet = FALSE)
}
\arguments{
\item{polygon_layer}{\code{\link[sp]{SpatialPolygons}}* object where the
polygon geometries represent any obstructions to fetch
calculations including the coastline, islands and/or 
exposed reefs.}

\item{site_layer}{\code{\link[sp]{SpatialPoints}}* object where the point 
geometries represent the site locations.}

\item{max_dist}{numeric. Maximum distance in kilometres (default 300). This 
will need to be scaled manually if the units for the CRS are 
not 'm'.}

\item{n_directions}{numeric. The number of fetch vectors to calculate per 
quadrant (default 9).}

\item{site_names}{character vector of the site names. If missing, the site 
names are taken from a column of the data associated with 
\code{site_layer} matching the regular expression 
\code{^[Nn]ames{0,1}}. If there is no such column, then 
default names are created ('Site 1', 'Site 2', ...).}

\item{quiet}{logical. Suppress diagnostic messages? (Default \code{FALSE}).}
}
\value{
Returns a \code{\link{Fetch}} object.
}
\description{
Wind fetch is the unobstructed length of water over which wind can blow, and
it is commonly used as a measure of exposure to wind and waves at coastal 
sites. The \code{fetch} function automatically calculates the wind fetch for
marine locations within the boundaries of the specified coastline layer.
This allows wind fetch to be calculated anywhere around the globe.
}
\details{
The function takes a \code{\link[sp]{SpatialPolygons-class}} object 
(\code{polygon_layer}) that represents the coastline, surrounding islands, 
and any other obstructions, and calculates the wind fetch for every specified 
direction. This is calculated for all the user-defined sites, that are 
represented as the point geometries in a 
\code{\link[sp]{SpatialPoints-class}} object.

The directions for which the wind fetch are calculated for each site are 
determined by the number of directions per quadrant (\code{n_directions}). 
The default value of 9 calculates 9 fetch vectors per quadrant (90 degrees), 
or equivalently, one fetch vector every 10 degrees. The first fetch vector is 
always calculated for the northerly direction (0/360 degrees).
}
\note{
At least one of the inputs to the \code{polygon_layer} or 
      \code{site_layer} arguments must be projected. If one of the inputs are 
      not projected, then it will be transformed to have the same projection 
      as the other. If both are projected, but do not have identical 
      coordinate reference systems (CRS) then \code{site_layer} will be 
      transformed to the same CRS as \code{polygon_layer}.
}
\examples{

# Create the polygon layer ----------------------------------------
#
# This is the layer that represents any obstacles that obstruct wind flow.

# Import map data for the Philippines.
philippines.df = ggplot2::map_data("world", region = "Philippines")

# Create a list for each separate polygon
philippines.list = split(philippines.df[, c("long", "lat")], 
                         philippines.df$group)
                         
library(sp)

philippines.Poly = lapply(philippines.list, Polygon)
philippines.Polys = list(Polygons(philippines.Poly, ID = "Philippines"))

# Include CRS information to make it a SpatialPolygons object
philippines.sp = SpatialPolygons(philippines.Polys, 
                                     proj4string = CRS("+init=epsg:4326"))

# Create the points layer ----------------------------------------
#
# The points layer represents the locations for which the wind fetch needs to
# be calculated.

# We need to calculate wind fetch for the following 3 sites:
sites.df = data.frame(lon = c(124.4824, 125.8473, 124.8416),
                      lat = c(9.167999, 9.751394, 11.478243),
                      site = c("Camiguin Island", "Bucas Grande Island",
                               "Talalora"))
                      
# Create the SpatialPoints object
sites.sp = SpatialPoints(sites.df[, 1:2], CRS("+init=epsg:4326"))

# Map projection -------------------------------------------------
#
# At least one of the polygon or points layers need to be projected to 
# calculate wind fetch.

# All these locations lie within the Philippines zone 5 / PRS92, that has
# WGS84 Bounds: 123.8000, 5.3000, 126.7000, 12.7500
# (http://spatialreference.org/ref/epsg/3125/)
# This suggests that this is a suitable map projection.
philippines.proj = spTransform(philippines.sp, "+init=epsg:3125")

# Calculate wind fetch -------------------------------------------
# 
# Calculate wind fetch at all the 3 locations for every 10 degrees on the
# compass rose, with a maximum distance for any fetch vector of 300 km.
my_fetch = fetch(philippines.proj, sites.sp, site_names = sites.df$site)
my_fetch

# Return only the summary data frame
summary(my_fetch)

# Transform the fetch vectors back to the original CRS
my_fetch_latlon = spTransform(my_fetch, proj4string(philippines.sp))

# Return the raw data in the original, lat/lon coordinates
my_fetch_latlon.df = as(my_fetch_latlon, "data.frame")
my_fetch_latlon.df

# Plot the wind fetch vectors ------------------------------------

# Plot the fetch vectors in the projected space...
plot(my_fetch, philippines.proj, axes = TRUE)

# ... or in the original coordinate reference system
plot(my_fetch, philippines.sp, axes = TRUE)

# Output to KML --------------------------------------------------
\dontrun{

# Save a KML file in the current working directory.
kml(my_fetch, colour = "white")
}
}
\seealso{
\code{\link[rgdal]{spTransform}} for methods on transforming map 
         projections and datum.

\code{\link[sp]{is.projected}} for checking whether a spatial object 
         is projected.

\code{\link{fetchR}} for an overview of this package with an
         extensive, reproducible example.

\code{\link{summary,Fetch-method}} for summarising the fetch lengths.
}
