#' Optimise Lineups
#'
#' Calculates optimal lineups for all franchises in the dataframe based on a table of lineup constraints.
#'
#' Lineup efficiency is the percentage of optimal/best-ball score that is used as the actual score - by default, the lineup efficiency for a team in non-best-ball settings is normally distributed around a mean of 77.5% and a standard deviation of 5%.
#'
#' @param roster_scores a dataframe as generated by `ffs_score_rosters()` - should contain columns like: projected_score, pos, and player_id
#' @param lineup_constraints a dataframe as generated by `ffscrapr::ff_starter_positions()` - should contain columns pos, min, max, and offense_starters
#' @param lineup_efficiency_mean the average lineup efficiency to use, defaults to 0.775
#' @param lineup_efficiency_sd the standard deviation of lineup efficiency, defaults to 0.05
#' @param best_ball a logical: FALSE will apply a lineup efficiency factor and TRUE uses optimal scores as actual scores, default = FALSE
#' @param parallel a logical: TRUE will run the optimization in parallel, requires the furrr and future packages as well as setting `future::plan()` in advance/externally. Default FALSE.
#'
#' @return a dataframe of what each team scored for each week
#'
#' @examples \donttest{
#' # cached examples
#' roster_scores <- .ffs_cache("roster_scores.rds")
#' lineup_constraints <- .ffs_cache("mfl_lineup_constraints.rds")
#'
#' ffs_optimise_lineups(roster_scores, lineup_constraints)
#'
#' }
#'
#' @seealso `vignette("Custom Simulations")` for example usage
#'
#' @export
ffs_optimise_lineups <- function(roster_scores,
                                 lineup_constraints,
                                 lineup_efficiency_mean = 0.775,
                                 lineup_efficiency_sd = 0.05,
                                 best_ball = FALSE,
                                 parallel = FALSE) {
  checkmate::assert_number(lineup_efficiency_mean, lower = 0, upper = 1)
  checkmate::assert_number(lineup_efficiency_sd, lower = 0, upper = 0.25)
  checkmate::assert_flag(best_ball)
  checkmate::assert_flag(parallel)

  checkmate::assert_data_frame(roster_scores)
  checkmate::assert_subset(
    c(
      "pos", "pos_rank", "league_id",
      "franchise_id", "franchise_name",
      "season", "week", "projected_score"
    ),
    names(roster_scores)
  )

  checkmate::assert_data_frame(lineup_constraints, any.missing = FALSE)
  checkmate::assert_subset(c("pos", "min", "max", "offense_starters"), names(lineup_constraints))

  if (!parallel) map <- purrr::map

  if (parallel && !requireNamespace("furrr", quietly = TRUE)) stop("Package {furrr} is required to run `ffs_optimise_lineups()` in parallel.", call. = FALSE)

  if (parallel) {
    map <- furrr::future_map

    if (inherits(future::plan(), "sequential")) {
      message("Parallel processing was specified but no future::plan() was found. Continuing sequentially.")
    }
  }

  lineup_constraints <- lineup_constraints %>%
    dplyr::filter(.data$pos %in% c("QB", "RB", "WR", "TE"))

  nest_data <- roster_scores %>%
    dplyr::left_join(
      lineup_constraints %>% dplyr::select("pos", "max"),
      by = "pos"
    ) %>%
    dplyr::filter(.data$pos_rank <= .data$max, .data$pos %in% lineup_constraints$pos) %>%
    dplyr::group_by(.data$league_id, .data$franchise_id, .data$franchise_name, .data$season, .data$week) %>%
    tidyr::nest() %>%
    dplyr::ungroup()

  if (best_ball) lineup_efficiency <- 1

  if (!best_ball) {
    lineup_efficiency <- stats::rnorm(nrow(nest_data),
      mean = lineup_efficiency_mean,
      sd = lineup_efficiency_sd
    )
  }

  optimal_scores <- nest_data %>%
    dplyr::mutate(
      optimals = map(.data$data, .ff_optimise_one_lineup, lineup_constraints),
      data = NULL
    ) %>%
    tidyr::unnest_wider("optimals") %>%
    dplyr::bind_cols(lineup_efficiency = lineup_efficiency) %>%
    dplyr::mutate(actual_score = .data$optimal_score * .data$lineup_efficiency)

  return(optimal_scores)
}
#' @rdname ffs_optimise_lineups
#' @export
ffs_optimize_lineups <- ffs_optimise_lineups

#' Optimise single lineup
#'
#' Optimises lineups for one franchise week at a time. Use purrr or loops to do more franchises/weeks/seasons
#'
#' @param franchise_scores a data frame of scores for one week and one franchise
#' @param lineup_constraints a data frame as created by `ffscrapr::ff_starter_positions()`
#'
#' @return a list including the optimal_score and the optimal_lineup tibble.
#'
#' @keywords internal
.ff_optimise_one_lineup <- function(franchise_scores, lineup_constraints) {
  min_req <- sum(lineup_constraints$min)

  player_ids <- c(franchise_scores$player_id, rep_len(NA_character_, min_req))
  player_scores <- c(franchise_scores$projected_score, rep_len(0, min_req))
  player_scores <- tidyr::replace_na(player_scores, 0)

  # binary - position identifiers

  pos_ids <- NULL

  for (i in lineup_constraints$pos) pos_ids <- c(pos_ids, as.integer(franchise_scores$pos == i), rep.int(1L, min_req))

  constraints_matrix <- matrix(
    c(
      pos_ids, # pos minimums
      pos_ids, # pos maximums
      as.integer(franchise_scores$pos %in% c("QB", "RB", "WR", "TE")), rep.int(1L, min_req)
    ), # total offensive starters
    nrow = nrow(lineup_constraints) * 2 + 1,
    byrow = TRUE
  )

  constraints_dir <- c(
    rep_len(">=", nrow(lineup_constraints)),
    rep_len("<=", nrow(lineup_constraints)),
    "<="
  )

  constraints_rhs <- c(
    lineup_constraints$min,
    lineup_constraints$max,
    lineup_constraints$offense_starters[[1]]
  )

  solve_lineup <- Rglpk::Rglpk_solve_LP(
    obj = player_scores,
    mat = constraints_matrix,
    dir = constraints_dir,
    rhs = constraints_rhs,
    types = rep("B", length(player_scores)),
    max = TRUE
  )

  optimals <- list(
    optimal_score = sum(player_scores * solve_lineup$solution),
    optimal_lineup = data.frame(
      player_id = player_ids[as.logical(solve_lineup$solution)],
      player_score = player_scores[as.logical(solve_lineup$solution)]
    )
  )

  return(optimals)
}
