#' Reformat R code while preserving blank lines and comments
#'
#' This function returns reformatted source code; it tries to preserve blank
#' lines and comments, which is different with \code{\link{parse}} and
#' \code{\link{deparse}}. It can also replace \code{=} with \code{<-} where
#' \code{=} means assignments, and reindent code by a specified number of spaces
#' (default is 4).
#'
#' This function helps the users to tidy up their source code in a sense that
#' necessary indents and spaces will be added, but comments will be preserved if
#' \code{keep.comment = TRUE}. See the references to know how this function
#' actually works.
#' @param source a character string: location of the source code (default to be
#'   the clipboard; this means we can copy the code to clipboard and use
#'   \code{tidy.souce()} without specifying the argument \code{source})
#' @param keep.comment whether to keep comments (\code{TRUE} by default)
#' @param keep.blank.line whether to keep blank lines (\code{TRUE} by default)
#' @param replace.assign whether to replace the assign operator \code{=} with
#'   \code{<-}
#' @param left.brace.newline whether to put the left brace \code{\{} to a new
#'   line (default \code{FALSE})
#' @param reindent.spaces number of spaces to indent the code (default 4)
#' @param output output to the console or a file using \code{\link{cat}}?
#' @param text an alternative way to specify the input: if it is \code{NULL},
#'   the function will read the source code from the \code{source} argument;
#'   alternatively, if \code{text} is a character vector containing the source
#'   code, it will be used as the input and the \code{source} argument will be
#'   ignored
#' @param width.cutoff passed to \code{\link{deparse}}: integer in [20, 500]
#'   determining the cutoff at which line-breaking is tried (default to be
#'   \code{getOption("width")})
#' @param ... other arguments passed to \code{\link{cat}}, e.g. \code{file}
#'   (this can be useful for batch-processing R scripts, e.g.
#'   \code{tidy.source(source = 'input.R', file = 'output.R')})
#' @return A list with components \item{text.tidy}{the reformatted code as a
#'   character vector} \item{text.mask}{the code containing comments, which are
#'   masked in assignments or with the weird operator}
#' @note Be sure to read the reference to know other limitations.
#' @author Yihui Xie <\url{http://yihui.name}> with substantial contribution
#'   from Yixuan Qiu <\url{http://yixuan.cos.name}>
#' @seealso \code{\link{parse}}, \code{\link{deparse}}, \code{\link{cat}}
#' @references \url{https://github.com/yihui/formatR/wiki/} (an introduction to
#'   this package, with examples and further notes)
#' @export
#' @example inst/examples/tidy.source.R
tidy.source = function(
  source = 'clipboard', keep.comment = getOption('keep.comment', TRUE),
  keep.blank.line = getOption('keep.blank.line', TRUE),
  replace.assign = getOption('replace.assign', FALSE),
  left.brace.newline = getOption('left.brace.newline', FALSE),
  reindent.spaces = getOption('reindent.spaces', 4),
  output = TRUE, text = NULL,
  width.cutoff = getOption('width'), ...
) {
  if (is.null(text)) {
    if (source == 'clipboard' && Sys.info()['sysname'] == 'Darwin') {
      source = pipe('pbpaste')
    }
  } else {
    source = textConnection(text); on.exit(close(source))
  }
  text = readLines(source, warn = FALSE)
  if (length(text) == 0L || all(grepl('^\\s*$', text))) {
    if (output) cat('\n', ...)
    return(list(text.tidy = text, text.mask = text))
  }
  if (keep.blank.line && R3) {
    one = paste(text, collapse = '\n') # record how many line breaks before/after
    n1 = attr(regexpr('^\n*', one), 'match.length')
    n2 = attr(regexpr('\n*$', one), 'match.length')
  }
  if (keep.comment) text = mask_comments(text, width.cutoff, keep.blank.line)
  text.mask = tidy_block(text, width.cutoff, replace.assign && length(grep('=', text)))
  text.tidy = if (keep.comment) unmask.source(text.mask) else text.mask
  text.tidy = reindent_lines(text.tidy, reindent.spaces)
  if (left.brace.newline) text.tidy = move_leftbrace(text.tidy)
  # restore new lines in the beginning and end
  if (keep.blank.line && R3) text.tidy = c(rep('', n1), text.tidy, rep('', n2))
  if (output) cat(paste(text.tidy, collapse = '\n'), '\n', ...)
  invisible(list(text.tidy = text.tidy, text.mask = text.mask))
}

## if you have variable names like this in your code, then you really beat me...
begin.comment = '.BeGiN_TiDy_IdEnTiFiEr_HaHaHa'
end.comment = '.HaHaHa_EnD_TiDy_IdEnTiFiEr'
pat.comment = sprintf('invisible\\("\\%s|\\%s"\\)', begin.comment, end.comment)
mat.comment = sprintf('invisible\\("\\%s([^"]*)\\%s"\\)', begin.comment, end.comment)
inline.comment = ' %InLiNe_IdEnTiFiEr%[ ]*"([ ]*#[^"]*)"'
blank.comment = sprintf('invisible("%s%s")', begin.comment, end.comment)

# wrapper around parse() and deparse()
tidy_block = function(text, width = getOption('width'), arrow = FALSE) {
  exprs = parse_only(text)
  if (length(exprs) == 0) return(character(0))
  exprs = if (arrow) replace_assignment(exprs) else as.list(exprs)
  sapply(exprs, function(e) paste(base::deparse(e, width), collapse = '\n'))
}

# Restore the real source code from the masked text
unmask.source = function(text.mask) {
  if (length(text.mask) == 0) return(text.mask)
  ## if the comments were separated into the next line, then remove '\n' after
  ##   the identifier first to move the comments back to the same line
  text.mask = gsub('%InLiNe_IdEnTiFiEr%[ ]*\n', '%InLiNe_IdEnTiFiEr%', text.mask)
  ## move 'else ...' back to the last line
  text.mask = gsub('\n\\s*else', ' else', text.mask)
  if (R3) {
    if (any(grepl('\\\\\\\\', text.mask)) && (any(grepl(mat.comment, text.mask)) ||
          any(grepl(inline.comment, text.mask)))) {
      m = gregexpr(mat.comment, text.mask)
      regmatches(text.mask, m) = lapply(regmatches(text.mask, m), restore_bs)
      m = gregexpr(inline.comment, text.mask)
      regmatches(text.mask, m) = lapply(regmatches(text.mask, m), restore_bs)
    }
  } else text.mask = restore_bs(text.mask)
  text.tidy = gsub(pat.comment, '', text.mask)
  # inline comments should be termined by $ or \n
  text.tidy = gsub(paste(inline.comment, '(\n|$)', sep = ''), '  \\1\\2', text.tidy)
  # the rest of inline comments should be appended by \n
  gsub(inline.comment, '  \\1\n', text.tidy)
}


#' Format the R scripts under a directory
#'
#' This function first looks for all the R scripts under a directory (using the
#' pattern \code{"[.][RrSsQq]$"}), then uses \code{\link{tidy.source}} to tidy
#' these scripts. The original scripts will be overwritten with reformatted code
#' if reformatting was successful. You may need to back up the original
#' directory first if you do not fully understand the tricks
#' \code{\link{tidy.source}} is using.
#' @param path the directory
#' @param recursive whether to recursively look for R scripts under \code{path}
#' @param ... other arguments to be passed to \code{\link{tidy.source}}
#' @return NULL
#' @author Yihui Xie <\url{http://yihui.name}>
#' @seealso \code{\link{tidy.source}}
#' @export
#' @examples
#' library(formatR)
#'
#' path = tempdir()
#' file.copy(system.file('demo', package = 'base'), path, recursive=TRUE)
#' tidy.dir(path, recursive=TRUE)
tidy.dir = function(path = '.', recursive = FALSE, ...) {
  flist = list.files(path, pattern = '[.][RrSsQq]$', full.names = TRUE, recursive = recursive)
  for (f in flist) {
    message('tidying ', f)
    try(tidy.source(f, file = f, ...))
  }
}
