#' Start building a data query
#' 
#' To download data from the ALA (or another atlas), one must construct a data 
#' query. This query tells the atlas API what data to download and return, as 
#' well as how it should be filtered. The `galah` package enables users to 
#' construct their data queries using piping syntax (i.e., `%>%` from `magrittr`, 
#' or `|>` from `base`). Building a data query using piping allows users to use 
#' functions like [search_taxa()], [galah_filter()], `[galah_select()]`, 
#' [galah_group_by()], and [galah_down_to()] to specify filters for their 
#' queries line-by-line, narrowing the results of their query. 
#' Users then finish their query with an `atlas_` function to identify which 
#' type of data they wish to download (i.e., [atlas_occurrences()], 
#' [atlas_counts()], [atlas_species()], [atlas_taxonomy()] or [atlas_media()]).
#' 
#' @param identify `data.frame`: generated by a call to [galah_identify()]
#' @param filter `data.frame`: generated by a call to [select_filters()]
#' @param select `data.frame`: generated by a call to [galah_select()]
#' @param geolocate `string`: generated by a call to [galah_geolocate()]
#' @param data_profile `string`: generated by a call to [galah_apply_profile()]
#' @param group_by `data.frame`: generated by a call to [galah_group_by()]
#' @param down_to `data.frame`: generated by a call to [galah_down_to()]
#' @param ... other function-specific request parameters
#' @return An object of class `data_request`.
#' 
#' @section Examples:
#' ```{r, child = "man/rmd/setup.Rmd"}
#' ```
#' 
#' Using `galah_call()` with pipes allows you to build, filter 
#' and download data queries the same way that you would wrangle your own data 
#' with `dplyr` and the `tidyverse`. Begin your query with `galah_call()`, then 
#' use either `%>%` from `magrittr` or `|>` from `base` to narrow/filter your 
#' results line-by-line.
#' 
#' Find number of records of *Aves* from 2001 to 2004 by year
#' 
#' ```{r, comment = "#>", collapse = TRUE, results = "hide", message = FALSE}
#' galah_call() |>
#'   galah_identify("Aves") |>
#'   galah_filter(year > 2000 & year < 2005) |>
#'   galah_group_by() |>
#'   atlas_counts()
#' ```
#'   
#' Download *Eolophus* records from 2001 to 2004
#' 
#' ```{r, comment = "#>", collapse = TRUE, results = "hide", eval = FALSE}
#' galah_config(email = "your-email@email.com")
#' 
#' galah_call() |>
#'    galah_identify("Eolophus") |>
#'   galah_filter(year > 2000 & year < 2005) |>
#'   atlas_occurrences()
#' ```
#' 
#' Find information for all species in *Cacatuidae* family
#' 
#' ```{r, comment = "#>", collapse = TRUE, results = "hide"}
#' galah_call() |>
#'   galah_identify("Cacatuidae") |>
#'   atlas_species()
#' ```
#' 
#' @export galah_call
galah_call <- function(identify = NULL, 
                       filter = NULL, 
                       select = NULL,
                       geolocate = NULL, 
                       data_profile = NULL,
                       group_by = NULL,
                       down_to = NULL,
                       ...){
  
  extra_args <- list(...)
  
  if(!is.null(identify)){ check_taxa_arg(identify) }
  check_call_args(filter, "filter")
  check_call_args(select, "select")
  check_call_args(geolocate, "geolocate")
  check_call_args(group_by, "group_by")
  check_call_args(data_profile, "data_profile")
  check_call_args(down_to, "down_to")
  
  request <- structure(c(list(identify = identify,
                              select = select, 
                              filter = filter,
                              geolocate = geolocate,
                              data_profile = data_profile,
                              group_by = group_by,
                              down_to = down_to),
                         extra_args),
                       class = "data_request")
  
  return(request)
}


check_call_args <- function(arg_supplied, arg_name, error_call = caller_env()){
  
  # check that, if an object is supplied, it is created by the correct function
  call_attr <- attr(arg_supplied, "call")
  has_call <- !is.null(call_attr)
  correct_call <- if(has_call){
      call_attr == paste0("galah_", arg_name)
    }else{
      FALSE
    }
  
  # if exists BUT doesn't have the correct attr, then abort
  if(!is.null(arg_supplied) && any(!c(has_call, correct_call))){
    abort(
      glue("`{arg_name}` argument requires an input generated by `galah_{arg_name}()`"),
      call = error_call)
  }
}


update_galah_call <- function(data_request, ...){
  dots <- list(...)
  result <- lapply(
    names(data_request), # i.e. for all slots in object of class `data_request`
    function(a){
      if(any(names(dots) == a)){ # object is present in `data_request`
        if(is.null(data_request[[a]])){ # slot in `data_request` is empty
          dots[[a]]
        }else{ # slot is filled
          if(is.null(dots[[a]])){ # if nothing has been supplied, retain source
            data_request[[a]]
          }else{ # both supplied and source contain data
            result <- switch(a,
              "identify" = {
                bind_unique_rows(data_request[[a]], dots[[a]], "identifier")
              },
              "filter" = {
                bind_unique_rows(data_request[[a]], dots[[a]], "query")
              },
              "select" = {
                bind_unique_rows(data_request[[a]], dots[[a]], "name")
              }, 
              # for below, we assume that in all other circumstances we 
              # simply pass the most recent result (i.e. overwrite)
              dots[[a]] # default
            )
            attr(result, "call") <- paste0("galah_", a) # add the `call` attr
            result
          }      
        }
      }else{ # if supplied object is not named in `data_request`
        data_request[[a]]
      }
    })
  names(result) <- names(data_request)

  # check if any names in `dots` have been missed from `results`
  missing_names <- !(names(dots) %in% names(result))
  if(any(missing_names)){
    result <- append(result, dots[missing_names])
  }
  class(result) <- "data_request"
  result
}


# 
bind_unique_rows <- function(x, y, column){
  result <- rbindlist(list(x, y), fill = TRUE)
  as_tibble(result[!duplicated(result[[column]]), ])
}



#' @keywords internal
#' @export
#' @rdname galah_call
# NOTE: use of `x` here is for consistency with `print()`; do not change
print.data_request <- function(x, ...){
  filled_slots <- !unlist(lapply(x, is.null))
  if(any(filled_slots)){
    cat("An object of type `data_request` containing:\n\n")
    print(as.list(x[filled_slots]))
  }else{
    cat("An empty object of type `data_request`")
  }
}


# function used in `galah_` and `search_` functions to determine 
# whether a quosure list contains an object of class `data_request`
# as its' first argument
detect_data_request <- function(dots){
  is_either <- (is_function_check(dots) | is_object_check(dots))
  if(length(is_either) < 1){
    is_either <- FALSE
  }else{
    is_either <- is_either[[1]]
  }
  if(is_either){
    eval_result <- try({eval_tidy(dots[[1]])}, silent = TRUE)
    if(inherits(eval_result, "try-error")){
      return(dots)
    }else if(inherits(eval_result, "data_request")){
      return(list(
        data_request = eval_result,
        dots = dots[-1]
      ))
    }else{
      return(dots)
    }
  }else{
    return(dots)
  }  
}