% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/display.R
\name{plot_raster}
\alias{plot_raster}
\title{Display raster data that have been read into a vector}
\usage{
plot_raster(
  data,
  xsize,
  ysize,
  nbands = 1,
  col_tbl = NULL,
  normalize = TRUE,
  minmax_def = NULL,
  minmax_pct_cut = NULL,
  col_map_fn = NULL,
  xlim = c(0, xsize),
  ylim = c(ysize, 0),
  interpolate = TRUE,
  asp = 1,
  axes = TRUE,
  main = "",
  xlab = "x",
  ylab = "y",
  xaxs = "i",
  yaxs = "i",
  legend = NULL,
  na_col = rgb(0, 0, 0, 0),
  ...
)
}
\arguments{
\item{data}{A numeric vector of pixel data to display, arranged in left to
right, top to bottom pixel order.}

\item{xsize}{The number of pixels along the x dimension in \code{data}.}

\item{ysize}{The number of pixels along the y dimension in \code{data}.}

\item{nbands}{The number of bands in \code{data}. Must be either 1 (grayscale) or
3 (RGB). For RGB, \code{data} are interleaved by band.}

\item{col_tbl}{A color table as a matrix or data frame with four columns.
Column 1 contains the numeric raster values, columns 2:4 contain the
intensities (between 0 and 1) of the red, green and blue primaries.}

\item{normalize}{Logical. \code{TRUE} to rescale pixel values so that their
range is \verb{[0,1]}, normalized to the full range of the pixel data by default
(\code{min(data)}, \code{max(data)}, per band). Ignored if \code{col_tbl} is used.
\code{normalize=FALSE} if a custom color mapping function is used that
operates on raw pixel values (see \code{col_map_fn} below).}

\item{minmax_def}{Normalize to user-defined min/max values (in terms of
the pixel data, per band). For single-band grayscale, a numeric vector of
length two containing min, max. For 3-band RGB, a numeric vector of length
six containing b1_min, b2_min, b3_min, b1_max, b2_max, b3_max.}

\item{minmax_pct_cut}{Normalize to a truncated range of the pixel data using
percentile cutoffs (removes outliers). A numeric vector of length two giving
the percentiles to use (e.g., \code{c(2, 98)}). Applied per band. Ignored if
\code{minmax_def} is used. Set \code{normalize=FALSE} if using a color mapping
function that operates on raw pixel values.}

\item{col_map_fn}{An optional color map function (default is
\code{grDevices::gray} for single-band data or \code{grDevices::rgb} for 3-band).
Ignored if \code{col_tbl} is used.}

\item{xlim}{Numeric vector of length two giving the x coordinate range. The
default uses pixel/line coordinates (\code{c(0, xsize)}).}

\item{ylim}{Numeric vector of length two giving the y coordinate range. The
default uses pixel/line coordinates (\code{c(ysize, 0)}).}

\item{interpolate}{Logical indicating whether to apply linear interpolation
to the image when drawing (default \code{TRUE}).}

\item{asp}{Numeric. The aspect ratio y/x (see \code{?plot.window}).}

\item{axes}{Logical. \code{TRUE} to draw axes (the default).}

\item{main}{The main title (on top).}

\item{xlab}{Title for the x axis (see \code{?title}).}

\item{ylab}{Title for the y axis (see \code{?title}).}

\item{xaxs}{The style of axis interval calculation to be used for the x axis
(see \code{?par}).}

\item{yaxs}{The style of axis interval calculation to be used for the y axis
(see \code{?par}).}

\item{legend}{Logical indicating whether to include a legend on the plot.
By deafult, a legend will be included if plotting single-band data without
\code{col_tbl} specified. Legend is not currently supported for color tables or
with 3-band RGB data.}

\item{na_col}{Color to use for \code{NA} as a 7- or 9-character hexadecimal code.
The default is transparent (\code{"#00000000"}, the return value of
\code{rgb(0,0,0,0)}).}

\item{...}{Other parameters to be passed to \code{plot.default()}.}
}
\description{
\code{plot_raster()} displays raster data using base graphics.
}
\note{
\code{plot_raster()} uses the function \code{graphics::rasterImage()} for plotting
which is not supported on some devices (see \code{?rasterImage}).

A reduced resolution overview for display can be read by setting \code{out_xsize}
and \code{out_ysize} smaller than the raster region specified by \code{xsize}, \code{ysize}
in calls to \code{GDALRaster$read()} or \code{read_ds()}.
The GDAL_RASTERIO_RESAMPLING configuration option can be defined to
override the default resampling to one of BILINEAR, CUBIC, CUBICSPLINE,
LANCZOS, AVERAGE or MODE.
}
\examples{
## Elevation
elev_file <- system.file("extdata/storml_elev.tif", package="gdalraster")
ds <- new(GDALRaster, elev_file, read_only=TRUE)
ncols <- ds$getRasterXSize()
nrows <- ds$getRasterYSize()
r <- read_ds(ds)
ds$close()

# grayscale
plot_raster(r, xsize=ncols, ysize=nrows, main="Storm Lake elevation (m)")

# color ramp
elev_pal <- c("#008435","#B4E34F","#F5D157","#CF983B","#B08153","#FFFFFF")
ramp <- scales::colour_ramp(elev_pal, alpha=FALSE)
plot_raster(r, xsize=ncols, ysize=nrows, col_map_fn=ramp,
            main="Storm Lake elevation (m)")

## Landsat band combination
b4_file <- system.file("extdata/sr_b4_20200829.tif", package="gdalraster")
b5_file <- system.file("extdata/sr_b5_20200829.tif", package="gdalraster")
b6_file <- system.file("extdata/sr_b6_20200829.tif", package="gdalraster")
band_files <- c(b6_file, b5_file, b4_file)

ds <- new(GDALRaster, b5_file, read_only=TRUE)
ncols <- ds$getRasterXSize()
nrows <- ds$getRasterYSize()
ds$close()

r <- vector("integer")
for (f in band_files) {
  ds <- new(GDALRaster, f, read_only=TRUE)
  r <- c(r, read_ds(ds))
  ds$close()
}

plot_raster(r, xsize=ncols, ysize=nrows, nbands=3,
            main="Landsat 6-5-4 (vegetative analysis)")

## LANDFIRE existing veg cover with colors from the CSV attribute table
evc_file <- system.file("extdata/storml_evc.tif", package="gdalraster")
evc_vat <- system.file("extdata/LF20_EVC_220.csv", package="gdalraster")
vat <- read.csv(evc_vat)
head(vat)
vat <- vat[,c(1,6:8)]

ds <- new(GDALRaster, evc_file, read_only=TRUE)
ncols <- ds$getRasterXSize()
nrows <- ds$getRasterYSize()

r <- read_ds(ds)
ds$close()
plot_raster(r, xsize=ncols, ysize=nrows, col_tbl=vat, interpolate=FALSE,
            main="Storm Lake Existing Vegetation Cover")
}
\seealso{
\code{\link[=GDALRaster]{GDALRaster$read()}}, \code{\link[=read_ds]{read_ds()}}, \code{\link[=set_config_option]{set_config_option()}}
}
