% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gdal_cli.R
\name{gdal_cli}
\alias{gdal_cli}
\alias{gdal_commands}
\alias{gdal_usage}
\alias{gdal_run}
\alias{gdal_alg}
\title{Convenience functions for using \dQuote{gdal} CLI algorithms}
\usage{
gdal_commands(contains = "", recurse = TRUE, cout = TRUE)

gdal_usage(cmd = NULL)

gdal_run(cmd, args)

gdal_alg(cmd = NULL, args = NULL, parse = TRUE)
}
\arguments{
\item{contains}{Optional character string for filtering output to certain
commands, e.g., \code{gdal_commands("vector")}.}

\item{recurse}{Logical value, \code{TRUE} to include all subcommands recursively
(the default). Set to \code{FALSE} to include only the top-level \dQuote{gdal}
commands (i.e., "raster", "vector", etc.)}

\item{cout}{Logical value, \code{TRUE} to print a list of commands along with
their descriptions and help URLS to the console (the default).}

\item{cmd}{A character string or character vector containing the path to the
algorithm, e.g., \code{"raster reproject"} or \code{c("raster", "reproject")}.
Defaults to \code{"gdal"}, the main entry point to CLI commands.}

\item{args}{Either a character vector or a named list containing input
arguments of the algorithm (see section \dQuote{Algorithm argument syntax}).}

\item{parse}{Logical value, \code{TRUE} to attempt parsing \code{args} if they are
given in \code{gdal_alg()} (the default). Set to \code{FALSE} to instantiate the
algorithm without parsing arguments. The \code{$parseCommandLineArgs()}
method on the returned object can be called to parse arguments and obtain
the result of that, with potentially useful error messages.}
}
\description{
This set of functions can be used to access and run GDAL utilities as
\dQuote{gdal} command line interface (CLI) algorithms.

\strong{Requires GDAL >= 3.11.3}

\strong{Experimental} (see "Development status")
}
\details{
These functions are convenient for accessing and running GDAL CLI algorithms
by way of the C++ exposed class \code{\link{GDALAlg}}. See the class
documentation for additional information (\code{?GDALAlg}).

\code{gdal_commands()} prints a list of commands and their descriptions to the
console, and returns (invisibly) a data frame with columns named \code{command},
\code{description} and \code{URL}. The \code{contains} argument can be used to filter the
output, e.g., \code{gdal_commands("vector")} to return only commands for working
with vector inputs.

\code{gdal_usage()} prints a help message to the console for a given command, or
for the root \code{"gdal"} entry point if called with no argument. No return
value, called for that side effect only.

\code{gdal_run()} executes a GDAL CLI algorithm and returns it as an object of
class \code{\link{GDALAlg}}. A list containing algorithm output(s) can be
accessed by calling the \code{$outputs()} (plural) method on the returned
object, or, more conveniently in most cases, by calling \code{$output()}
(singular) to return the the single output value when there is only one.
After assigning the output, or otherwise completing work with the \code{GDALAlg}
object, its \code{$release()} method can be called to close datasets and
free resources.

\code{gdal_alg()} instantiates and returns an object of class \code{\link{GDALAlg}}
without running it. Passing argument values to the requested CLI algorithm is
optional. This function may be useful (with or without argument values) for
obtaining algorithm properties with the returned object's \code{$info()}
method, obtaining properties of algorithm arguments
(\code{$argInfo(arg_name)}), or obtaining algorithm usage as a
JSON-formatted string (\code{$usageAsJSON()}).
This function is simply an alternative to calling the \code{new()} constructor
for class \code{GDALAlg}. Executing the returned algorithm is optional by calling
the object's \code{$run()} method (assuming argument values were given).
}
\note{
Commands do not require the leading \code{"gdal"} root node. They may begin
with a top-level command (e.g., \code{"raster"}, \code{"vector"}, etc.).

When using argument names as the element names of a list, the underscore
character can be optionally substituted for the dash characters that are
used in some names. This avoids having to surround names in backticks when
they are used to access list elements in the form \code{args$arg_name} (the form
\code{args[["arg-name"]]} also works).
}
\section{Algorithm argument syntax}{

Arguments are given in \R as a character vector or named list, but
otherwise syntax basically matches the GDAL specification for arguments as
they are given on the command line. Those specifications are listed here
along with some amendments regarding the character vector and named list
formats. Programmatic usage also allows passing and receiving datasets as
objects (i.e., \code{GDALRaster} or \code{GDALVector}), in addition to dataset names
(e.g., filename, URL, database connection string).
\itemize{
\item Commands accept one or several positional arguments, typically for dataset
names (or, in \R, as \code{GDALRaster} or \code{GDALVector} datasets). The order is
input(s) first, output last. Positional arguments can also be specified as
named arguments, if preferred to avoid any ambiguity.
\item Named arguments have:
\itemize{
\item at least one "long" name, preceded by two dash characters when specified
on the command line,
\item optionally, auxiliary long names (i.e., aliases),
\item and optionally a one-letter short name, preceded by a single dash
character on the command line, e.g., \verb{-f, --of, --format, --output-format <OUTPUT-FORMAT>}
}
\item Boolean arguments are specified by just specifying the argument name. In
\R list format, the named element must be assigned a value of logical \code{TRUE}.
\item Arguments that require a value are specified like:
\itemize{
\item \verb{-f VALUE} for one-letter short names.
\item \verb{--format VALUE} or \code{--format=VALUE} for long names.
\item in a named list, this would look like:\cr
\code{arg_list$format <- VALUE}
}
\item Some arguments can be multi-valued. Some of them require all values to be
packed together and separated with comma. This is, e.g., the case of:
\verb{--bbox <BBOX> Clipping bounding box as xmin,ymin,xmax,ymax}, e.g.,
\verb{--bbox=2.1,49.1,2.9,49.9}.
\item Others accept each value to be preceded by a new mention of the argument
name, e.g., \verb{--co COMPRESS=LZW --co TILED=YES}. For that one, if the value
of the argument does not contain commas, the packed form is also accepted:
\verb{--co COMPRESS=LZW,TILED=YES}. Note that repeated mentions of an argument
are possible in the character vector format for argument input, whereas
arguments given in named list format must used argument long names as the
list element names, and the packed format for the values (which can be a
character vector or numeric vector of values).
\item Named arguments can be placed before or after positional arguments.
}
}

\section{Development status}{

The GDAL Command Line Interface Modernization was first introduced in the
\href{https://github.com/OSGeo/gdal/releases/tag/v3.11.0}{GDAL 3.11.0 release}
(2025-05-09). The GDAL project states that the new CLI \dQuote{is
provisionally provided as an alternative interface to GDAL and OGR command
line utilities. The project reserves the right to modify, rename,
reorganize, and change the behavior until it is officially frozen via PSC
vote in a future major GDAL release.... Your usage of it should have no
expectation of compatibility until that time.}
(\url{https://gdal.org/en/latest/programs/#gdal-application})

Initial bindings to enable programmatic use of the new CLI algorithms from \R
were added in \pkg{gdalraster} 2.2.0, and will evolve over future releases.
The bindings are considered experimental until the upstream API is declared
stable. Breaking changes in minor version releases are possible until then.
}

\examples{
\dontshow{if (gdal_version_num() >= gdalraster:::.GDALALG_MIN_GDAL) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
## top-level gdal commands
gdal_commands(recurse = FALSE)

## convert storml_elev.tif to GeoPackage raster
gdal_commands("convert")

gdal_usage("raster convert")

f_tif <- system.file("extdata/storml_elev.tif", package="gdalraster")
f_gpkg <- file.path(tempdir(), "storml_elev.gpkg")

args <- c("--overwrite", f_tif, f_gpkg)
(alg <- gdal_run("raster convert", args))

(ds <- alg$output())

alg$release()

plot_raster(ds, legend = TRUE)

ds$close()
unlink(f_gpkg)

## get help for vector commands
gdal_usage("vector")

## clip a vector layer by a bounding box
gdal_usage("vector clip")

f <- system.file("extdata/ynp_fires_1984_2022.gpkg", package="gdalraster")
f_clip <- file.path(tempdir(), "ynp_fires_clip.gpkg")

# some multi-valued arguments require all values packed and comma separated
# e.g., --bbox <BBOX>
bb <- c(469686, 11442, 544070, 85508)
bb <- paste(bb, collapse = ",")

args <- c("--bbox", bb, "--overwrite", f, f_clip)
(alg <- gdal_run("vector clip", args))

(lyr <- alg$output())

lyr$bbox()

lyr$getFeatureCount()

lyr$close()
alg$release()
unlink(f_clip)

## rasterize a vector layer and return output as a GDALRaster object
gdal_usage("vector rasterize")

f_out = file.path(tempdir(), "ynp_fire_year.tif")

# arguments in list format
args <- list(input = f,
             sql = "SELECT * FROM mtbs_perims ORDER BY ig_year",
             attribute_name = "ig_year",
             output = f_out,
             overwrite = TRUE,
             creation_option = c("TILED=YES", "COMPRESS=DEFLATE"),
             resolution = c(90, 90),
             output_data_type = "Int16",
             init = -32767,
             nodata = -32767)

(alg <- gdal_run("vector rasterize", args))

(ds <- alg$output())

alg$release()

pal <- scales::viridis_pal(end = 0.8, direction = -1)(6)
ramp <- scales::colour_ramp(pal)
plot_raster(ds, legend = TRUE, col_map_fn = ramp, na_col = "#d9d9d9",
            main = "YNP Fires 1984-2022 - Most Recent Burn Year")

ds$close()
deleteDataset(f_out)

## "pipeline" syntax
# "raster pipeline" example 2 from:
# https://gdal.org/en/latest/programs/gdal_raster_pipeline.html
# serialize the command to reproject a GTiff file into GDALG format, and
# then later read the GDALG file
# GDAL Streamed Algorithm format:
# https://gdal.org/en/stable/drivers/raster/gdalg.html

gdal_usage("raster pipeline")

f_tif <- system.file("extdata/storml_elev.tif", package="gdalraster")
f_out <- file.path(tempdir(), "storml_elev_epsg_32100.gdalg.json")

args <- c("read", "--input", f_tif, "!",
          "reproject", "--dst-crs=EPSG:32100", "!",
          "write", "--output", f_out, "--overwrite")

alg <- gdal_run("raster pipeline", args)
alg$release()

# content of the .gdalg.json file
readLines(f_out, warn = FALSE) |> writeLines()

(ds <- new(GDALRaster, f_out))

plot_raster(ds, legend = TRUE)

ds$close()
unlink(f_out)
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=GDALAlg]{GDALAlg-class}}

\dQuote{gdal} Command Line Interface (CLI):\cr
\url{https://gdal.org/en/stable/programs/index.html}

Using \dQuote{gdal} CLI algorithms from R:\cr
\url{https://usdaforestservice.github.io/gdalraster/articles/use-gdal-cli-from-r.html}
}
