% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convenience-functions.R
\name{me_diag}
\alias{me_diag}
\title{Measurement error model diagnostics}
\source{
Donegan, Connor and Chun, Yongwan and Griffith, Daniel A. (2021). ``Modeling community health with areal data: Bayesian inference with survey standard errors and spatial structure.'' \emph{Int. J. Env. Res. and Public Health} 18 (13): 6856. DOI: 10.3390/ijerph18136856 Data and code: \url{https://github.com/ConnorDonegan/survey-HBM}.
}
\usage{
me_diag(
  fit,
  varname,
  shape,
  probs = c(0.025, 0.975),
  plot = TRUE,
  mc_style = c("scatter", "hist"),
  size = 0.25,
  index = 0,
  style = c("W", "B"),
  w = shape2mat(shape, match.arg(style)),
  binwidth = function(x) 0.5 * sd(x)
)
}
\arguments{
\item{fit}{A \code{geostan_fit} model object as returned from a call to one of the \code{geostan::stan_*} functions.}

\item{varname}{Name of the modeled variable (a character string, as it appears in the model formula).}

\item{shape}{An object of class \code{sf} or another spatial object coercible to \code{sf} with \code{sf::st_as_sf}.}

\item{probs}{Lower and upper quantiles of the credible interval to plot.}

\item{plot}{If \code{FALSE}, return a list of \code{ggplot}s and a \code{data.frame} with the raw data values alongside a posterior summary of the modeled variable.}

\item{mc_style}{Character string indicating how to plot the Moran coefficient for the delta values: if \code{mc = "scatter"}, then \code{\link[geostan]{moran_plot}} will be used with the marginal residuals; if \code{mc = "hist"}, then a histogram of Moran coefficient values will be returned, where each plotted value represents the degree of residual autocorrelation in a draw from the join posterior distribution of delta values.}

\item{size}{Size of points and lines, passed to \code{geom_pointrange}.}

\item{index}{Integer value; use this if you wish to identify observations with the largest \code{n=index} absolute Delta values; data on the top \code{n=index} observations ordered by absolute Delta value will be printed to the console and the plots will be labeled with the indices of the identified observations.}

\item{style}{Style of connectivity matrix; if \code{w} is not provided, \code{style} is passed to \code{\link[geostan]{shape2mat}} and defaults to "W" for row-standardized.}

\item{w}{An optional spatial connectivity matrix; if not provided, one will be created using \code{\link[geostan]{shape2mat}}.}

\item{binwidth}{A function with a single argument that will be passed to the \code{binwidth} argument in \code{\link[ggplot2]{geom_histogram}}. The default is to set the width of bins to \code{0.5 * sd(x)}.}
}
\value{
A grid of spatial diagnostic plots for measurement error models comparing the raw observations to the posterior distribution of the true values. Includes a point-interval plot of raw values and modeled values; a Moran scatter plot for \code{delta = z - x} where \code{z} are the survey estimates and \code{x} are the modeled values; and a map of the delta values (take at their posterior means).
}
\description{
Visual diagnostics for spatial measurement error models.
}
\examples{
\donttest{
library(sf)
data(georgia)
## binary adjacency matrix
A <- shape2mat(georgia, "B")
## prepare data for the CAR model, using WCAR specification
cars <- prep_car_data(A, style = "WCAR")
## provide list of data for the measurement error model
ME <- prep_me_data(se = data.frame(ICE = georgia$ICE.se),
                   car_parts = cars)
## sample from the prior probability model only, including the ME model
fit <- stan_glm(log(rate.male) ~ ICE,
                ME = ME,
                data = georgia, 
                prior_only = TRUE,
                iter = 800, # for speed only
                chains = 2, # for speed only
                refresh = 0 # silence some printing
                )

## see ME diagnostics
me_diag(fit, "ICE", georgia)
## see index values for the largest (absolute) delta values
 ## (differences between raw estimate and the posterior mean)
me_diag(fit, "ICE", georgia, index = 3)
}
}
\seealso{
\code{\link[geostan]{sp_diag}}, \code{\link[geostan]{moran_plot}}, \code{\link[geostan]{mc}}, \code{\link[geostan]{aple}}
}
