% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GGExtensions.R
\name{stat_compare}
\alias{stat_compare}
\title{Add Mean Comparison for Groups to a ggplot}
\usage{
stat_compare(
  mapping = NULL,
  data = NULL,
  position = "identity",
  ...,
  nudge = 0,
  start = NULL,
  breaks = NULL,
  labels = NULL,
  cutoff = NULL,
  method = NULL,
  ref_group = NULL,
  tip_length = 0.02,
  parametric = FALSE,
  correction = "none",
  panel_indep = FALSE,
  method_args = NULL,
  comparisons = NULL,
  step_increase = 0.1,
  inherit.aes = TRUE
)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[ggplot2:aes]{aes()}}. If specified and
\code{inherit.aes = TRUE} (the default), it is combined with the default mapping
at the top level of the plot. You must supply \code{mapping} if there is no plot
mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[ggplot2:ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[ggplot2:fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data. A \code{function} can be created
from a \code{formula} (e.g. \code{~ head(.x, 10)}).}

\item{position}{A position adjustment to use on the data for this layer. This
can be used in various ways, including to prevent overplotting and
improving the display. The \code{position} argument accepts the following:
\itemize{
\item The result of calling a position function, such as \code{position_jitter()}.
This method allows for passing extra arguments to the position.
\item A string naming the position adjustment. To give the position as a
string, strip the function name of the \code{position_} prefix. For example,
to use \code{position_jitter()}, give the position as \code{"jitter"}.
\item For more information and other ways to specify the position, see the
\link[ggplot2:layer_positions]{layer position} documentation.
}}

\item{...}{additional arguments passed on to \code{\link[=geom_bracket]{geom_bracket()}}.}

\item{nudge}{\code{numeric}, the nudge of start position in fraction of scale range.}

\item{start}{\code{numeric}, the bracket start position. Defaults to the maximum value of \code{y}.}

\item{breaks}{\code{numeric}, the breaks for p-value labels, like c(0, 0.001, 0.01, 0.05, 1).}

\item{labels}{\code{character}, the labels for p-value breaks, like c("\emph{\strong{", "}", "}", "ns").}

\item{cutoff}{\code{numeric}, the cutoff for p-value, labels above this value will be removed.}

\item{method}{\code{function}, the method for the test; it should support formula interface and return a list with components \code{p.value} and \code{method} (name).}

\item{ref_group}{\code{character}, the reference group for comparison. other groups will be compared to this group.}

\item{tip_length}{\code{numeric}, the length of the bracket tips in fraction of scale range.}

\item{parametric}{\code{logical}, whether to use parametric test (t-test, One-way ANOVA) or non-parametric test (Wilcoxon rank sum test, Kruskal-Wallis test). Applicable only when \code{method} is NULL.}

\item{correction}{\code{character}, the method for p-value adjustment; options include \link[stats:p.adjust]{p.adjust.methods} with "\code{none}" as the default.}

\item{panel_indep}{\code{logical}, whether to correct the p-value only at the panel level. If \code{FALSE}, the p-value will be corrected at the layer level.}

\item{method_args}{\code{list}, additional arguments to be passed to the test method.}

\item{comparisons}{\code{list}, a list of comparisons to be made. Each element should contain two groups to be compared.}

\item{step_increase}{\code{numeric}, the step increase in fraction of scale range for every additional comparison, in order to avoid overlapping brackets.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[ggplot2:borders]{borders()}}.}
}
\value{
\code{LayerInstance}, a layer that can be added to a ggplot.
}
\description{
Add group mean comparisons to a ggplot. The comparisons can be performed using the t-test, Wilcoxon rank-sum test, one-way ANOVA, or Kruskal-Wallis test.
}
\details{
Usually you do not need to specify the test method, you only need to tell \code{stat_compare()} whether you want to perform a parametric test or a nonparametric test, and \code{stat_compare()} will automatically choose the appropriate test method based on your data.
For comparisons between two groups, the p-value is calculated by t-test (parametric) or Wilcoxon rank sum test (nonparametric). For comparisons among more than two groups, the p-value is calculated by One-way ANOVA (parametric) or Kruskal-Wallis test (nonparametric).
}
\section{Aesthetics}{

\itemize{
\item required: \code{x}, \code{y}
}
}

\section{Computed variables}{

\itemize{
\item \strong{\code{p}}: p-value of the test.
\item \strong{\code{q}}: adjusted p-value of the test.
\item \strong{\code{label}}: the label of the bracket.
\item \strong{\code{method}}: the method name of the test.
\item \strong{\code{xmin}}, \strong{\code{xmax}}, \strong{\code{ymin}}, \strong{\code{ymax}}: the position of the bracket.
}
}

\examples{
\dontshow{if (interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
p <- ggplot(mpg, aes(class, displ, color = class)) +
  geom_boxplot(show.legend = FALSE) +
  theme_test()

# Global comparison: Each x has only one group.
p + stat_compare()
# If you just want to display text, you can set parameters "bracket" to FALSE.
p + stat_compare(bracket = FALSE)
# If you want to display the test method, you can do this.
p + stat_compare(aes(label = after_stat(sprintf("\%s: \%s", method, label))))

# Comparison between two groups: specify a reference group.
p + stat_compare(ref_group = "minivan")
# If you only want to display the p-value less or equal to 0.01, you can do this.
p + stat_compare(ref_group = "minivan", cutoff = 0.01)
# if you want to display the significance level, you can do this.
p + stat_compare(ref_group = "minivan", breaks = c(0, 0.001, 0.01, 0.05, 1))

# Comparison between two groups: specify the comparison group.
p + stat_compare(tip_length = 0.05,
                 step_increase = 0,
                 comparisons = list(c("compact", "midsize"), c("pickup", "suv")),
                 arrow = grid::arrow(type = "closed", length = unit(0.1, "inches")))
                 # Yeah, this supports adding arrows.

# Within-group (grouped by the x-axis) population comparison.
ggplot(mpg, aes(drv, displ, fill = class)) +
  geom_boxplot() +
  stat_compare() +
  stat_compare(aes(group = drv), nudge = 0.1, color = "gray") + # add global comparison
  theme_test()

# Better adaptation to faceting.
ggplot(mpg, aes(drv, displ)) +
  geom_boxplot() +
  stat_compare(comparisons = combn(unique(mpg$drv), 2, simplify = FALSE)) +
  facet_grid(cols = vars(class), scales = "free") +
  theme_test()

# P-value correction
p <- ggplot(mpg, aes(class, displ)) +
  geom_boxplot() +
  facet_grid(cols = vars(cyl), scales = "free") +
  theme_test()
# Layer-level P-value correction
p + stat_compare(ref_group = 1, correction = "fdr")
# Panel-level P-value correction
p + stat_compare(ref_group = 1, correction = "fdr", panel_indep = TRUE)
\dontshow{\}) # examplesIf}
}
\author{
HMU-WH
}
