% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smooth.R
\name{smooth_density}
\alias{smooth_density}
\alias{smooth_bounded}
\alias{smooth_unbounded}
\title{Smooth dot positions in a dotplot using a kernel density estimator ("density dotplots")}
\usage{
smooth_bounded(
  x,
  density = "bounded",
  bounds = c(NA, NA),
  bounder = "cooke",
  trim = FALSE,
  ...
)

smooth_unbounded(x, density = "unbounded", trim = FALSE, ...)
}
\arguments{
\item{x}{a numeric vector}

\item{density}{Density estimator to use for smoothing. One of:
\itemize{
\item A function which takes a numeric vector and returns a list with elements
\code{x} (giving grid points for the density estimator) and \code{y} (the
corresponding densities). \pkg{ggdist} provides a family of functions
following this format, including \code{\link[=density_unbounded]{density_unbounded()}} and
\code{\link[=density_bounded]{density_bounded()}}.
\item A string giving the suffix of a function name that starts with \code{"density_"};
e.g. \code{"bounded"} for \verb{[density_bounded()]}.
}}

\item{bounds}{length-2 vector of min and max bounds. If a bound is \code{NA}, then
that bound is estimated from the data using the method specified by \code{bounder}.}

\item{bounder}{Method to use to find missing (\code{NA}) \code{bounds}. A function that
takes a numeric vector of values and returns a length-2 vector of the estimated
lower and upper bound of the distribution. Can also be a string giving the
suffix of the name of such a function that starts with \code{"bounder_"}. Useful
values include:
\itemize{
\item \code{"cdf"}: Use the CDF of the the minimum and maximum order statistics of the
sample to estimate the bounds. See \code{\link[=bounder_cdf]{bounder_cdf()}}.
\item \code{"cooke"}: Use the method from Cooke (1979); i.e. method 2.3 from Loh (1984).
See \code{\link[=bounder_cooke]{bounder_cooke()}}.
\item \code{"range"}: Use the range of \code{x} (i.e the \code{min} or \code{max}). See \code{\link[=bounder_range]{bounder_range()}}.
}}

\item{trim}{Should the density estimate be trimmed to the bounds of the data?}

\item{...}{Arguments passed to the density estimator specified by \code{density}.}
}
\value{
A numeric vector of \code{length(x)}, where each entry is a smoothed version of
the corresponding entry in \code{x}.

If \code{x} is missing, returns a partial application of itself. See \link{automatic-partial-functions}.
}
\description{
Smooths \code{x} values using a density estimator, returning new \code{x} of the same
length. Can be used with a dotplot (e.g. \code{\link{geom_dots}}\code{(smooth = ...)}) to create
"density dotplots".
Supports \link[=automatic-partial-functions]{automatic partial function application}.
}
\details{
Applies a kernel density estimator (KDE) to \code{x}, then uses weighted quantiles
of the KDE to generate a new set of \code{x} values with smoothed values. Plotted
using a dotplot (e.g. \code{geom_dots(smooth = "bounded")} or
\verb{geom_dots(smooth = smooth_bounded(...)}), these values create a variation on
a "density dotplot" (Zvinca 2018).

Such plots are recommended \strong{only} in very
large sample sizes where precise positions of individual values are not
particularly meaningful. In small samples, normal dotplots should generally
be used.

Two variants are supplied by default:
\itemize{
\item \code{smooth_bounded()}, which uses \code{\link[=density_bounded]{density_bounded()}}.
Passes the \code{bounds} arguments to the estimator.
\item \code{smooth_unbounded()}, which uses \code{\link[=density_unbounded]{density_unbounded()}}.
}

It is generally recommended to pick the smooth based on the known bounds of
your data, e.g. by using \code{smooth_bounded()} with the \code{bounds} parameter if
there are finite bounds, or \code{smooth_unbounded()} if both bounds are infinite.
}
\examples{

library(ggplot2)

set.seed(1234)
x = rnorm(1000)

# basic dotplot is noisy
ggplot(data.frame(x), aes(x)) +
  geom_dots()

# density dotplot is smoother, but does move points (most noticeable
# in areas of low density)
ggplot(data.frame(x), aes(x)) +
  geom_dots(smooth = "unbounded")

# you can adjust the kernel and bandwidth...
ggplot(data.frame(x), aes(x)) +
  geom_dots(smooth = smooth_unbounded(kernel = "triangular", adjust = 0.5))

# for bounded data, you should use the bounded smoother
x_beta = rbeta(1000, 0.5, 0.5)

ggplot(data.frame(x_beta), aes(x_beta)) +
  geom_dots(smooth = smooth_bounded(bounds = c(0, 1)))

}
\references{
Zvinca, Daniel. "In the pursuit of diversity in data visualization. Jittering data to access details."
\url{https://www.linkedin.com/pulse/pursuit-diversity-data-visualization-jittering-access-daniel-zvinca/}.
}
\seealso{
Other dotplot smooths: 
\code{\link{smooth_discrete}()},
\code{\link{smooth_none}()}
}
\concept{dotplot smooths}
