% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gginteraction.R
\name{gginteraction}
\alias{gginteraction}
\title{Get marginal effects for two-way interactions from models}
\usage{
gginteraction(model, mdrt.values = "minmax", swap.pred = FALSE,
  ci.lvl = 0.95, ...)
}
\arguments{
\item{model}{A fitted model object, or a list of model objects. Any model
that is supported by the \CRANpkg{effects}-package should work.}

\item{mdrt.values}{Indicates which values of the moderator variable should be
used to calculate marginal effects of the interaction.
\describe{
  \item{\code{"minmax"}}{(default) minimum and maximum values (lower and upper bounds) of the moderator are used to plot the interaction between independent variable and moderator.}
  \item{\code{"meansd"}}{uses the mean value of the moderator as well as one standard deviation below and above mean value to plot the effect of the moderator on the independent variable.}
  \item{\code{"zeromax"}}{is similar to the \code{"minmax"} option, however, \code{0} is always used as minimum value for the moderator. This may be useful for predictors that don't have an empirical zero-value, but absence of moderation should be simulated by using 0 as minimum.}
  \item{\code{"quart"}}{calculates and uses the quartiles (lower, median and upper) of the moderator value.}
  \item{\code{"all"}}{uses all values of the moderator variable. Note that this option only applies to \code{type = "eff"}, for numeric moderator values.}
}}

\item{swap.pred}{Logical, if \code{TRUE}, the predictor (defining the x-position)
and the moderator (defining the groups) in an interaction are
swapped. By default, the first interaction term is considered as
moderator and the second term is used to define the x-position.}

\item{ci.lvl}{Numeric, the level of the confidence intervals. For \code{ggpredict()},
use \code{ci.lvl = NA}, if confidence intervals should not be calculated
(for instance, due to computation time).}

\item{...}{Further arguments passed down to \code{\link[effects]{effect}}.}
}
\value{
A tibble (with \code{ggeffects} class attribute) with consistent data columns:
        \describe{
          \item{\code{x}}{the values of the model predictor to which the effect pertains, used as x-position in plots.}
          \item{\code{predicted}}{the predicted values, used as y-position in plots.}
          \item{\code{conf.low}}{the lower bound of the confidence interval for the predicted values.}
          \item{\code{conf.high}}{the upper bound of the confidence interval for the predicted values.}
          \item{\code{group}}{the name of \code{x}, used as grouping-aesthetics in plots.}
        }
}
\description{
\code{gginteraction()} computes marginal effects of interaction terms.
               It internally calls \code{\link[effects]{effect}} and
               puts the result into tidy data frames.
}
\note{
\code{gginteraction()} only compute marginal effects for interaction terms,
      in particular two-way interactions. Use \code{\link{ggeffect}} for
      marginal effects for simple model terms. Or use \code{\link{ggpredict}}
      for predictions from models with three-way interactions.
}
\examples{
data(efc)
efc$c172code <- sjmisc::to_factor(efc$c172code)
fit <- lm(barthtot ~ c12hour + c161sex + c172code * neg_c_7, data = efc)
gginteraction(fit)

# this would give the same results
ggpredict(fit, terms = c("neg_c_7", "c172code"))

library(ggplot2)
ggplot(gginteraction(fit), aes(x, predicted, colour = group)) +
  geom_line()

dat <- gginteraction(fit)
ggplot(dat, aes(x, predicted, colour = group)) +
  geom_line() +
  labs(
    colour = get_legend_title(dat),
    x = get_x_title(dat),
    y = get_y_title(dat),
    title = get_title(dat)
  )  +
  scale_color_manual(
    values = c("red", "green", "blue"),
    labels = get_legend_labels(dat)
  )

# use continuous term on x-axis, but use values mean +/- sd as groups
dat <- gginteraction(fit, mdrt.values = "meansd", swap.pred = TRUE)
ggplot(dat, aes(x, predicted, colour = group)) + geom_line()

}
