#' Make numeric column of a data.frame to factor
#'
#' @param data a data.frame
#' @param colnames Column names to be converted
#' @param maxfactorno maximun unique value of column
#' @export
num2factorDf=function(data,colnames,maxfactorno=6){
    if(!is.null(colnames)){
        for(i in 1 :length(colnames)){
            name<-colnames[i]
            if(is.numeric(data[[name]])&(length(unique(data[[name]]))<=maxfactorno)){
                data[[name]]<-factor(data[[name]])
            }

        }
    }
    data
}

#' Unselect numeric column of a data.frame
#'
#' @param data a data.frame
#' @param colnames Column names to be converted
#' @param maxfactorno maximun unique value of column
#' @export
unselectNumeric=function(data,colnames,maxfactorno=6){
        result=c()
        if(!is.null(colnames)){
                for(i in 1 :length(colnames)){
                        name<-colnames[i]
                        if(is.numeric(data[[name]])&(length(unique(data[[name]]))>maxfactorno)){
                                result=c(result,name)
                        }

                }
        }
        result<-setdiff(colnames,result)
        result
}


#'Make a regression equation of a model
#'
#'@param model A model of class "lm" or"glm" or"loess"
#'@param digits integer indicating the number of decimal places
#'@export
makeEq=function(model,digits=2) {

       # model=lm(mpg~wt,data=mtcars);digits=2
       # str(model)
       # summary(model)


    intercept<-model$coef[1]
    slope<-model$coef[2]
    (eq=paste0("method=",class(model)[1]))
    if("glm" %in% class(model)){
        eq=paste0(eq,"<br>y=1/(1+exp(-(",round(slope,digits),"x",ifelse(intercept>=0,"+","-"),abs(round(intercept,digits)),")))")
    }
    else if("lm" %in% class(model)){
        eq=paste0(eq,"<br>y = ",round(intercept,2),ifelse(slope>=0,"+","-"),abs(round(slope,2)),"x")
        if(length(model$coef)>2){
            for(i in 3:length(model$coef))
            eq=paste0(eq,ifelse(model$coef[i]>=0,"+","-"),abs(round(model$coef[i],2)),"x^",(i-1))
        }
    }
    if((class(model)[1]=="lm")|(class(model)[1]=="glm")){
    pval=NULL
    if(nrow(model$model)>1) pval=summary(model)$coef[2,4]

    if(!is.null(pval)) {
            if(pval<0.0001) eq=paste0(eq,"<br>p < 0.0001")
            else eq=paste0(eq,"<br>p = ",sprintf("%0.4f",pval))
    }
    }
    eq

}

#'Make a data.fram of yhat with a model
#'
#'@param model A model of class "lm" or"glm" or"loess"
#'@param x A optional vector of explanatory variable
#'@param n number of observations.
model2df=function(model,x=NULL,n=100){
    #x=NULL;n=100;model=mgcv::gam(status~nodes,data=colon);digits=2
    #str(model)

    (dfnames=names(attr(model$terms,"dataClasses")))
    if(is.null(x)){
        if(class(model)[1] %in% c("glm","lm","gam")) x<-model$model[2]

         else x<-model$x
        # if(is.null(n)) n=length(x)
        (xmin=min(x,na.rm=TRUE))
        (xmax=max(x,na.rm=TRUE))
        newx=seq(xmin,xmax,length.out=n)
         #newx=x

    } else {
        newx=x
    }
    data1=data.frame(newx)
    colnames(data1)=dfnames[2]

    if("glm" %in% class(model)){
        newy=predict(model,data1,type="response")
    } else {
        newy=predict(model,data1)
    }

    data2=data.frame(newy,newx)
    #str(data2)
    colnames(data2)=dfnames[1:2]
    data2
}

#' Make an interactive scatterplot with regression line(s)
#'@param data a data.frame
#'@param mapping Set of aesthetic mappings created by aes or aes_.
#'@param smooth Logical. Add regression lines to the scatter plot
#'@param se Logical. display confidence interval around linear regression? (TRUE by default)
#'@param method smoothing method (function) to use, eg. "lm", "glm", "gam", "loess", "rlm"
#'@param formula formula to use in smoothing function, eg. y ~ x, y ~ poly(x, 2), y ~ log(x)
#'@param fullrange should the fit span the full range of the plot, or just the data
#'@param level level of confidence interval to use (0.95 by default)
#'@param maxfactorno An integer. Maximum unique number of a numeric vector treated as a factor
#'@param digits integer indicating the number of decimal places
#'@param tooltip A character string of column name be included in tooltip. Default value is NULL
#'@param interactive A logical value. If TRUE, an interactive plot will be returned
#'@param title The text for plot title
#'@param subtitle The text for plot subtitle
#'@param caption The text for plot caption
#'@param ... other arguments passed on to geom_point
#'@export
#'@importFrom ggplot2 ggplot stat_smooth aes aes_string position_jitter facet_wrap labs
#'@importFrom ggiraph ggiraph geom_point_interactive geom_path_interactive
#'@importFrom mgcv gam
#'@importFrom plyr dlply splat
#'@importFrom stats as.formula predict
#'@examples
#'require(ggplot2)
#'require(ggiraph)
#'require(plyr)
#'ggPoints(aes(x=wt,y=mpg,color=carb),data=mtcars,interactive=TRUE,maxfactno=3)
ggPoints=function(data,mapping, smooth=TRUE,
                  se=TRUE,method="auto",formula=y~x, fullrange=FALSE,level=0.95,
                  maxfactorno=6,digits=2,title=NULL,subtitle=NULL,caption=NULL,
                  tooltip=NULL,interactive=FALSE,...) {


         # data=mtcars;mapping=aes(x=wt,y=mpg,color=carb);smooth=TRUE
         # se=TRUE;method="auto";formula=y~x; fullrange=FALSE;level=0.95;
         # maxfactorno=3;digits=2;
         # tooltip=NULL;interactive=TRUE;

    #formula=y~x

    if(method=="auto"){
        if(nrow(data)<1000) {
                method<-"loess"
        } else {
            method<-"gam"
            if(identical(formula,y~x)) formula<-y~s(x,bs="cs")
        }
    }


    #str(mapping)
    xname=paste(mapping[["x"]])

    yname=paste(mapping[["y"]])
    facetname=NULL
    if("facet" %in% names(mapping)) facetname<-paste(mapping[["facet"]])
    colorname=NULL
    if("colour" %in% names(mapping)) colorname<-paste(mapping[["colour"]])


    (formulas=as.character(formula))
    (right=gsub("x",xname,formulas[3]))
    if(grepl("poly(",right,fixed=TRUE)==TRUE){
        (equation=unlist(strsplit(right,",")))
        number=as.numeric(unlist(strsplit(equation[2],")")))
        temp=xname
        for(i in 2:number){
            temp=paste0(temp,"+I(",xname,"^",i,")")
        }
        right=temp
    }
    myformula=paste0(yname,"~",right)
    myformula


    temp=paste0("function(",yname,",",xname,",...){",method,"(",myformula)
    if(method=="glm") temp=paste0(temp,",family=binomial")
    (temp=paste0(temp,")}"))

    (grepModel=eval(parse(text=temp)))


    (groupname=setdiff(names(mapping),c("x","y")))

    (groupvar=paste(mapping[groupname]))
    if(length(groupvar)>0)  {
            data=num2factorDf(data,groupvar,maxfactorno=maxfactorno)
            groupvar=unselectNumeric(data,groupvar,maxfactorno=maxfactorno)
            if(!is.null(facetname)) groupvar=c(groupvar,facetname)
    }
    data$id=1:nrow(data)
    if(is.null(tooltip)) data$tooltip=data$id
    else data$tooltip=data[[tooltip]]
    data$tooltip=paste0(data$tooltip,"<br>",xname,"=",data[[xname]],"<br>",yname,"=",data[[yname]])
    data

    p<-ggplot(data,mapping)

    if(smooth) {
        if(method=="glm") {
        p<-p+stat_smooth(method='glm',formula=formula,method.args=list(family='binomial'),se=se,fullrange=fullrange)
        p<-p+geom_point_interactive(aes_string(data_id="id",tooltip="tooltip"),
                                    position=position_jitter(width=0.3,height=0.06),alpha=0.5,...)
        } else {
        p<-p+ stat_smooth(method=method,formula=formula,se=se,fullrange=fullrange)
        p<-p+geom_point_interactive(aes_string(data_id="id",tooltip="tooltip"),...)
        #p<-p+geom_point_interactive(aes_string(data_id="id",tooltip="tooltip"))

        }
    }
    p
    p1<-p
    colorname
    if(length(groupvar)<1){
        model=splat(grepModel)(data)

        (equation=makeEq(model))

        data2=model2df(model)
        data2$id=1
        data2$tooltip=equation



        if(is.null(colorname)){
        p<-p+geom_path_interactive(aes_string(x=xname,y=yname,data_id="id",tooltip="tooltip"),
                                   color="blue",data=data2)
        } else {
                data2[[colorname]]=min(data[[colorname]])

        p<-p+geom_path_interactive(aes_string(x=xname,y=yname,data_id="id",tooltip="tooltip"),
                                   color="blue",data=data2)
        }
    } else {

        ## exclude subset data with nrow<1
         data<-ddply(data,groupvar,function(d) {if(nrow(d)>1) d else NULL})

        data

        (models=dlply(data,groupvar,splat(grepModel)))
        strata=attr(models,"split_labels")
        for(i in 1:length(models)){


            if(fullrange){
                (xmin=min(data[[xname]],na.rm=TRUE))
                (xmax=max(data[[xname]],na.rm=TRUE))
                newx=seq(xmin,xmax,length.out=100)

                data2=model2df(models[[i]],x=newx)
            } else{
                data2=model2df(models[[i]])
            }

            data2$tooltip=makeEq(models[[i]])
            data2$id=i
            if(!is.null(strata)){
                for(j in 1:ncol(strata)){
                    data2=cbind(data2,j=strata[i,j])
                    colnames(data2)[ncol(data2)]=names(strata)[j]
                    data2$tooltip=paste0("for ",names(strata)[j],"=",strata[i,j],"<br>",data2$tooltip)
                }
            }
            if(is.null(colorname)){
                    p<-p+geom_path_interactive(aes_string(x=xname,y=yname,data_id="id",tooltip="tooltip"),
                                       color="blue",data=data2)
            } else{
                    p<-p+geom_path_interactive(aes_string(x=xname,y=yname,data_id="id",tooltip="tooltip",
                                                          color=colorname),
                                               data=data2)
            }
        }
    }

     p<-p+labs(title=title,subtitle=subtitle,caption=caption)

        #str(data2)
    if(!is.null(facetname)) {
        formula1=as.formula(paste0("~",facetname))
        p<-p+facet_wrap(formula1)
    }

    if(interactive){
        tooltip_css <- "background-color:white;font-style:italic;padding:10px;border-radius:10px 20px 10px 20px;"
        #hover_css="fill-opacity=.3;cursor:pointer;stroke:gold;"
        hover_css="r:4px;cursor:pointer;stroke-width:6px;"
        selected_css = "fill:#FF3333;stroke:black;"
        p<-ggiraph(code=print(p),tooltip_extra_css=tooltip_css,tooltip_opacity=.75,
                   zoom_max=10,hover_css=hover_css,selected_css=selected_css)
    }
    p
}



