% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coord-sf.R, R/geom-sf.R, R/stat-sf.R
\docType{data}
\name{CoordSf}
\alias{CoordSf}
\alias{coord_sf}
\alias{ggsf}
\alias{GeomSf}
\alias{geom_sf}
\alias{geom_sf_label}
\alias{geom_sf_text}
\alias{StatSf}
\alias{stat_sf}
\title{Visualise sf objects}
\usage{
coord_sf(
  xlim = NULL,
  ylim = NULL,
  expand = TRUE,
  crs = NULL,
  default_crs = NULL,
  datum = sf::st_crs(4326),
  label_graticule = waiver(),
  label_axes = waiver(),
  lims_method = c("cross", "box", "orthogonal", "geometry_bbox"),
  ndiscr = 100,
  default = FALSE,
  clip = "on"
)

geom_sf(
  mapping = aes(),
  data = NULL,
  stat = "sf",
  position = "identity",
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE,
  ...
)

geom_sf_label(
  mapping = aes(),
  data = NULL,
  stat = "sf_coordinates",
  position = "identity",
  ...,
  parse = FALSE,
  nudge_x = 0,
  nudge_y = 0,
  label.padding = unit(0.25, "lines"),
  label.r = unit(0.15, "lines"),
  label.size = 0.25,
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE,
  fun.geometry = NULL
)

geom_sf_text(
  mapping = aes(),
  data = NULL,
  stat = "sf_coordinates",
  position = "identity",
  ...,
  parse = FALSE,
  nudge_x = 0,
  nudge_y = 0,
  check_overlap = FALSE,
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE,
  fun.geometry = NULL
)

stat_sf(
  mapping = NULL,
  data = NULL,
  geom = "rect",
  position = "identity",
  na.rm = FALSE,
  show.legend = NA,
  inherit.aes = TRUE,
  ...
)
}
\arguments{
\item{xlim, ylim}{Limits for the x and y axes. These limits are specified
in the units of the default CRS. By default, this means projected coordinates
(\code{default_crs = NULL}). How limit specifications translate into the exact
region shown on the plot can be confusing when non-linear or rotated coordinate
systems are used as the default crs. First, different methods can be preferable
under different conditions. See parameter \code{lims_method} for details. Second,
specifying limits along only one direction can affect the automatically generated
limits along the other direction. Therefore, it is best to always specify limits
for both x and y. Third, specifying limits via position scales or \code{xlim()}/\code{ylim()}
is strongly discouraged, as it can result in data points being dropped from the plot even
though they would be visible in the final plot region.}

\item{expand}{If \code{TRUE}, the default, adds a small expansion factor to
the limits to ensure that data and axes don't overlap. If \code{FALSE},
limits are taken exactly from the data or \code{xlim}/\code{ylim}.}

\item{crs}{The coordinate reference system (CRS) into which all data should
be projected before plotting. If not specified, will use the CRS defined
in the first sf layer of the plot.}

\item{default_crs}{The default CRS to be used for non-sf layers (which
don't carry any CRS information) and scale limits. The default value of
\code{NULL} means that the setting for \code{crs} is used. This implies that all
non-sf layers and scale limits are assumed to be specified in projected
coordinates. A useful alternative setting is \code{default_crs = sf::st_crs(4326)},
which means x and y positions are interpreted as longitude and latitude,
respectively, in the World Geodetic System 1984 (WGS84).}

\item{datum}{CRS that provides datum to use when generating graticules.}

\item{label_graticule}{Character vector indicating which graticule lines should be labeled
where. Meridians run north-south, and the letters \code{"N"} and \code{"S"} indicate that
they should be labeled on their north or south end points, respectively.
Parallels run east-west, and the letters \code{"E"} and \code{"W"} indicate that they
should be labeled on their east or west end points, respectively. Thus,
\code{label_graticule = "SW"} would label meridians at their south end and parallels at
their west end, whereas \code{label_graticule = "EW"} would label parallels at both
ends and meridians not at all. Because meridians and parallels can in general
intersect with any side of the plot panel, for any choice of \code{label_graticule} labels
are not guaranteed to reside on only one particular side of the plot panel. Also,
\code{label_graticule} can cause labeling artifacts, in particular if a graticule line
coincides with the edge of the plot panel. In such circumstances, \code{label_axes} will
generally yield better results and should be used instead.

This parameter can be used alone or in combination with \code{label_axes}.}

\item{label_axes}{Character vector or named list of character values
specifying which graticule lines (meridians or parallels) should be labeled on
which side of the plot. Meridians are indicated by \code{"E"} (for East) and
parallels by \code{"N"} (for North). Default is \code{"--EN"}, which specifies
(clockwise from the top) no labels on the top, none on the right, meridians
on the bottom, and parallels on the left. Alternatively, this setting could have been
specified with \code{list(bottom = "E", left = "N")}.

This parameter can be used alone or in combination with \code{label_graticule}.}

\item{lims_method}{Method specifying how scale limits are converted into
limits on the plot region. Has no effect when \code{default_crs = NULL}.
For a very non-linear CRS (e.g., a perspective centered
around the North pole), the available methods yield widely differing results, and
you may want to try various options. Methods currently implemented include \code{"cross"}
(the default), \code{"box"}, \code{"orthogonal"}, and \code{"geometry_bbox"}. For method \code{"cross"},
limits along one direction (e.g., longitude) are applied at the midpoint of the
other direction (e.g., latitude). This method avoids excessively large limits for
rotated coordinate systems but means that sometimes limits need to be expanded a
little further if extreme data points are to be included in the final plot region.
By contrast, for method \code{"box"}, a box is generated out of the limits along both directions,
and then limits in projected coordinates are chosen such that the entire box is
visible. This method can yield plot regions that are too large. Finally, method
\code{"orthogonal"} applies limits separately along each axis, and method
\code{"geometry_bbox"} ignores all limit information except the bounding boxes of any
objects in the \code{geometry} aesthetic.}

\item{ndiscr}{Number of segments to use for discretising graticule lines;
try increasing this number when graticules look incorrect.}

\item{default}{Is this the default coordinate system? If \code{FALSE} (the default),
then replacing this coordinate system with another one creates a message alerting
the user that the coordinate system is being replaced. If \code{TRUE}, that warning
is suppressed.}

\item{clip}{Should drawing be clipped to the extent of the plot panel? A
setting of \code{"on"} (the default) means yes, and a setting of \code{"off"}
means no. In most cases, the default of \code{"on"} should not be changed,
as setting \code{clip = "off"} can cause unexpected results. It allows
drawing of data points anywhere on the plot, including in the plot margins. If
limits are set via \code{xlim} and \code{ylim} and some data points fall outside those
limits, then those data points may show up in places such as the axes, the
legend, the plot title, or the plot margins.}

\item{mapping}{Set of aesthetic mappings created by \code{\link[=aes]{aes()}} or
\code{\link[=aes_]{aes_()}}. If specified and \code{inherit.aes = TRUE} (the
default), it is combined with the default mapping at the top level of the
plot. You must supply \code{mapping} if there is no plot mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[=ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[=fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data. A \code{function} can be created
from a \code{formula} (e.g. \code{~ head(.x, 10)}).}

\item{stat}{The statistical transformation to use on the data for this
layer, as a string.}

\item{position}{Position adjustment, either as a string, or the result of
a call to a position adjustment function.}

\item{na.rm}{If \code{FALSE}, the default, missing values are removed with
a warning. If \code{TRUE}, missing values are silently removed.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.

You can also set this to one of "polygon", "line", and "point" to
override the default legend.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[=borders]{borders()}}.}

\item{...}{Other arguments passed on to \code{\link[=layer]{layer()}}. These are
often aesthetics, used to set an aesthetic to a fixed value, like
\code{colour = "red"} or \code{size = 3}. They may also be parameters
to the paired geom/stat.}

\item{parse}{If \code{TRUE}, the labels will be parsed into expressions and
displayed as described in \code{?plotmath}.}

\item{nudge_x}{Horizontal and vertical adjustment to nudge labels by.
Useful for offsetting text from points, particularly on discrete scales.
Cannot be jointly specified with \code{position}.}

\item{nudge_y}{Horizontal and vertical adjustment to nudge labels by.
Useful for offsetting text from points, particularly on discrete scales.
Cannot be jointly specified with \code{position}.}

\item{label.padding}{Amount of padding around label. Defaults to 0.25 lines.}

\item{label.r}{Radius of rounded corners. Defaults to 0.15 lines.}

\item{label.size}{Size of label border, in mm.}

\item{fun.geometry}{A function that takes a \code{sfc} object and returns a \code{sfc_POINT} with the
same length as the input. If \code{NULL}, \code{function(x) sf::st_point_on_surface(sf::st_zm(x))}
will be used. Note that the function may warn about the incorrectness of
the result if the data is not projected, but you can ignore this except
when you really care about the exact locations.}

\item{check_overlap}{If \code{TRUE}, text that overlaps previous text in the
same layer will not be plotted. \code{check_overlap} happens at draw time and in
the order of the data. Therefore data should be arranged by the label
column before calling \code{geom_text()}. Note that this argument is not
supported by \code{geom_label()}.}

\item{geom}{The geometric object to use display the data}
}
\description{
This set of geom, stat, and coord are used to visualise simple feature (sf)
objects. For simple plots, you will only need \code{geom_sf()} as it
uses \code{stat_sf()} and adds \code{coord_sf()} for you. \code{geom_sf()} is
an unusual geom because it will draw different geometric objects depending
on what simple features are present in the data: you can get points, lines,
or polygons.
For text and labels, you can use \code{geom_sf_text()} and \code{geom_sf_label()}.
}
\section{Geometry aesthetic}{

\code{geom_sf()} uses a unique aesthetic: \code{geometry}, giving an
column of class \code{sfc} containing simple features data. There
are three ways to supply the \code{geometry} aesthetic:
\itemize{
\item Do nothing: by default \code{geom_sf()} assumes it is stored in
the \code{geometry} column.
\item Explicitly pass an \code{sf} object to the \code{data} argument.
This will use the primary geometry column, no matter what it's called.
\item Supply your own using \code{aes(geometry = my_column)}
}

Unlike other aesthetics, \code{geometry} will never be inherited from
the plot.
}

\section{CRS}{

\code{coord_sf()} ensures that all layers use a common CRS. You can
either specify it using the \code{crs} param, or \code{coord_sf()} will
take it from the first layer that defines a CRS.
}

\section{Combining sf layers and regular geoms}{

Most regular geoms, such as \code{\link[=geom_point]{geom_point()}}, \code{\link[=geom_path]{geom_path()}},
\code{\link[=geom_text]{geom_text()}}, \code{\link[=geom_polygon]{geom_polygon()}} etc. will work fine with \code{coord_sf()}. However
when using these geoms, two problems arise. First, what CRS should be used
for the x and y coordinates used by these non-sf geoms? The CRS applied to
non-sf geoms is set by the \code{default_crs} parameter, and it defaults to
\code{NULL}, which means positions for non-sf geoms are interpreted as projected
coordinates in the coordinate system set by the \code{crs} parameter. This setting
allows you complete control over where exactly items are placed on the plot
canvas, but it may require some understanding of how projections work and how
to generate data in projected coordinates. As an alternative, you can set
\code{default_crs = sf::st_crs(4326)}, the World Geodetic System 1984 (WGS84).
This means that x and y positions are interpreted as longitude and latitude,
respectively. You can also specify any other valid CRS as the default CRS for
non-sf geoms.

The second problem that arises for non-sf geoms is how straight lines
should be interpreted in projected space when \code{default_crs} is not set to \code{NULL}.
The approach \code{coord_sf()} takes is to break straight lines into small pieces
(i.e., segmentize them) and then transform the pieces into projected coordinates.
For the default setting where x and y are interpreted as longitude and latitude,
this approach means that horizontal lines follow the parallels and vertical lines
follow the meridians. If you need a different approach to handling straight lines,
then you should manually segmentize and project coordinates and generate the plot
in projected coordinates.
}

\examples{
if (requireNamespace("sf", quietly = TRUE)) {
nc <- sf::st_read(system.file("shape/nc.shp", package = "sf"), quiet = TRUE)
ggplot(nc) +
  geom_sf(aes(fill = AREA))

# If not supplied, coord_sf() will take the CRS from the first layer
# and automatically transform all other layers to use that CRS. This
# ensures that all data will correctly line up
nc_3857 <- sf::st_transform(nc, 3857)
ggplot() +
  geom_sf(data = nc) +
  geom_sf(data = nc_3857, colour = "red", fill = NA)

# Unfortunately if you plot other types of feature you'll need to use
# show.legend to tell ggplot2 what type of legend to use
nc_3857$mid <- sf::st_centroid(nc_3857$geometry)
ggplot(nc_3857) +
  geom_sf(colour = "white") +
  geom_sf(aes(geometry = mid, size = AREA), show.legend = "point")

# You can also use layers with x and y aesthetics. To have these interpreted
# as longitude/latitude you need to set the default CRS in coord_sf()
ggplot(nc_3857) +
  geom_sf() +
  annotate("point", x = -80, y = 35, colour = "red", size = 4) +
  coord_sf(default_crs = sf::st_crs(4326))

# Thanks to the power of sf, a geom_sf nicely handles varying projections
# setting the aspect ratio correctly.
library(maps)
world1 <- sf::st_as_sf(map('world', plot = FALSE, fill = TRUE))
ggplot() + geom_sf(data = world1)

world2 <- sf::st_transform(
  world1,
  "+proj=laea +y_0=0 +lon_0=155 +lat_0=-90 +ellps=WGS84 +no_defs"
)
ggplot() + geom_sf(data = world2)

# To add labels, use geom_sf_label().
ggplot(nc_3857[1:3, ]) +
   geom_sf(aes(fill = AREA)) +
   geom_sf_label(aes(label = NAME))
}
}
\seealso{
\code{\link[=stat_sf_coordinates]{stat_sf_coordinates()}}
}
\keyword{datasets}
