#' Draw Points on a ggplot Object
#'
#' @description
#' This function overlays points to a ggplot object using data from a CSV file generated by the ggsem Shiny app or any custom dataset.
#' Points can be styled with various shapes, colors, sizes, and orientations.
#' @param p A ggplot object to which the points will be added.
#' @param points_data A data frame containing information about the points to be drawn. The expected columns include:
#' \itemize{
#'   \item \code{x}, \code{y}: Coordinates of the point.
#'   \item \code{shape}: Shape of the point (\code{"circle"}, \code{"square"}, \code{"triangle"}, \code{"rectangle"}, \code{"oval"}, or \code{"diamond"}).
#'   \item \code{color}: Fill color of the point (hexadecimal color code).
#'   \item \code{size}: Size of the point.
#'   \item \code{border_color}: Border color of the point (hexadecimal color code).
#'   \item \code{border_width}: Width of the border.
#'   \item \code{alpha}: Transparency of the point (numeric, 0 to 1).
#'   \item \code{width_height_ratio}: Ratio of width to height (for shapes like rectangles and ovals).
#'   \item \code{orientation}: Rotation angle of the point in degrees (for shapes like rectangles and diamonds).
#' }
#' @param zoom_level Numeric. Adjusts the size of the points relative to the plot. Default is \code{1}.
#'
#' @return
#' A ggplot object with the specified points added.
#' @export
#'
#' @examples
#' library(ggplot2)
#'
#' points_data <- data.frame(
#' x = 20, y = 20, shape = 'rectangle', color = '#D0C5ED', size = 50,
#' border_color = '#9646D4', border_width = 2, alpha = 1, width_height_ratio = 1.6, orientation = 45,
#' lavaan = FALSE, lavaan = FALSE, network = FALSE, locked = FALSE
#' )
#'
#' p <- ggplot()
#'
#' draw_points(p, points_data, zoom_level = 1.2) +
#' scale_x_continuous(limits = c(0,50)) +
#' scale_y_continuous(limits = c(0,50))
#'
draw_points <- function(p, points_data, zoom_level = 1) {
  if (nrow(points_data) > 0) {
    points_data$color <- sapply(points_data$color, valid_hex)
    points_data$border_color <- sapply(points_data$border_color, valid_hex)
    points_data$shape <- sapply(points_data$shape, valid_shape)
    points_data$alpha <- sapply(points_data$alpha, valid_alpha)

    if (length(points_data$color) != nrow(points_data)) {
      points_data$color <- rep(points_data$color[1], nrow(points_data))
    }

    if (length(points_data$border_width) != nrow(points_data)) {
      points_data$border_width <- rep(points_data$border_width[1], nrow(points_data))
    }

    for (i in 1:nrow(points_data)) {
      shape <- points_data$shape[i]
      adjusted_stroke <- points_data$border_width[i] / zoom_level
      adjusted_width <- (points_data$size[i] / 3)
      adjusted_height <- adjusted_width / ifelse(!is.null(points_data$width_height_ratio[i]),
                                                 points_data$width_height_ratio[i], 1
      )

      min_size_factor <- 0.25
      scale_factor <- sqrt(2)
      if (shape %in% c("circle")) {
        adjusted_width <- points_data$size[i] / scale_factor  * min_size_factor
        adjusted_height <- adjusted_width
      } else if (shape == "square") {
        adjusted_width <- points_data$size[i] * sqrt(2) * min_size_factor
        adjusted_height <- adjusted_width
      } else if (shape == "triangle") {
        adjusted_width <- points_data$size[i] * sqrt(4 / sqrt(3)) * min_size_factor
        adjusted_height <- adjusted_width * sqrt(3) / 2
      } else if (shape == "rectangle") {
        width_height_ratio <- ifelse(!is.null(points_data$width_height_ratio[i]),
                                     points_data$width_height_ratio[i], 1)
        adjusted_height <- points_data$size[i] * min_size_factor
        adjusted_width <- adjusted_height * width_height_ratio
      } else if (shape == "diamond") {
        width_height_ratio <- ifelse(!is.null(points_data$width_height_ratio[i]),
                                     points_data$width_height_ratio[i], 1)

        adjusted_height <- points_data$size[i] * 1.4 * sqrt(1.5) * min_size_factor
        adjusted_width <- adjusted_height * width_height_ratio
      } else if (shape == "oval") {
        width_height_ratio <- ifelse(!is.null(points_data$width_height_ratio[i]),
                                     points_data$width_height_ratio[i], 1)
        adjusted_height <- points_data$size[i] * min_size_factor / scale_factor
        adjusted_width <- adjusted_height * width_height_ratio
      }

      if (shape == "circle") {
        t <- seq(0, 2 * pi, length.out = 100)
        circle_coords <- data.frame(
          x = points_data$x[i] + adjusted_width * cos(t),
          y = points_data$y[i] + adjusted_height * sin(t)
        )
        p <- p + annotate(
          "polygon",
          x = circle_coords$x,
          y = circle_coords$y,
          fill = points_data$color[i],
          colour = points_data$border_color[i],
          size = adjusted_stroke,
          alpha = points_data$alpha[i]
        )
      } else if (shape == "triangle") {
        triangle_coords <- data.frame(
          x = c(
            points_data$x[i],
            points_data$x[i] - adjusted_width / 2,
            points_data$x[i] + adjusted_width / 2
          ),
          y = c(
            points_data$y[i] + adjusted_height / 2,
            points_data$y[i] - adjusted_height / 2,
            points_data$y[i] - adjusted_height / 2
          )
        )
        rotated_coords <- rotate_coords(triangle_coords$x, triangle_coords$y, points_data$orientation[i],
                                        cx = points_data$x[i], cy = points_data$y[i]
        )
        p <- p + annotate(
          "polygon",
          x = rotated_coords$x,
          y = rotated_coords$y,
          fill = points_data$color[i],
          colour = points_data$border_color[i],
          size = adjusted_stroke,
          alpha = points_data$alpha[i]
        )
      } else if (shape == "square") {
        rect_coords <- data.frame(
          x = c(
            points_data$x[i] - adjusted_width / 2,
            points_data$x[i] + adjusted_width / 2,
            points_data$x[i] + adjusted_width / 2,
            points_data$x[i] - adjusted_width / 2
          ),
          y = c(
            points_data$y[i] - adjusted_height / 2,
            points_data$y[i] - adjusted_height / 2,
            points_data$y[i] + adjusted_height / 2,
            points_data$y[i] + adjusted_height / 2
          )
        )
        rotated_coords <- rotate_coords(rect_coords$x, rect_coords$y, points_data$orientation[i],
                                        cx = points_data$x[i], cy = points_data$y[i]
        )
        p <- p + annotate(
          "polygon",
          x = rotated_coords$x,
          y = rotated_coords$y,
          fill = points_data$color[i],
          colour = points_data$border_color[i],
          size = adjusted_stroke,
          alpha = points_data$alpha[i]
        )
      } else if (shape == "oval") {
        t <- seq(0, 2 * pi, length.out = 100)
        oval_coords <- data.frame(
          x = points_data$x[i] + adjusted_width * cos(t),
          y = points_data$y[i] + adjusted_height * sin(t)
        )
        rotated_coords <- rotate_coords(oval_coords$x, oval_coords$y, points_data$orientation[i],
                                        cx = points_data$x[i], cy = points_data$y[i]
        )
        p <- p + annotate(
          "polygon",
          x = rotated_coords$x,
          y = rotated_coords$y,
          fill = points_data$color[i],
          colour = points_data$border_color[i],
          size = adjusted_stroke,
          alpha = points_data$alpha[i]
        )
      } else if (shape == "rectangle") {
        rect_coords <- data.frame(
          x = c(
            points_data$x[i] - adjusted_width / 2,
            points_data$x[i] + adjusted_width / 2,
            points_data$x[i] + adjusted_width / 2,
            points_data$x[i] - adjusted_width / 2
          ),
          y = c(
            points_data$y[i] - adjusted_height / 2,
            points_data$y[i] - adjusted_height / 2,
            points_data$y[i] + adjusted_height / 2,
            points_data$y[i] + adjusted_height / 2
          )
        )

        rotated_coords <- rotate_coords(rect_coords$x, rect_coords$y, points_data$orientation[i],
                                        cx = points_data$x[i], cy = points_data$y[i]
        )

        p <- p + annotate(
          "polygon",
          x = rotated_coords$x,
          y = rotated_coords$y,
          fill = points_data$color[i],
          colour = points_data$border_color[i],
          size = adjusted_stroke,
          alpha = points_data$alpha[i]
        )
      } else {
        # diamond
        diamond_coords <- data.frame(
          x = c(
            points_data$x[i],
            points_data$x[i] - adjusted_width / 2,
            points_data$x[i],
            points_data$x[i] + adjusted_width / 2
          ),
          y = c(
            points_data$y[i] + adjusted_height / 2,
            points_data$y[i],
            points_data$y[i] - adjusted_height / 2,
            points_data$y[i]
          )
        )
        rotated_coords <- rotate_coords(diamond_coords$x, diamond_coords$y, points_data$orientation[i],
                                        cx = points_data$x[i], cy = points_data$y[i]
        )
        p <- p + annotate(
          "polygon",
          x = rotated_coords$x,
          y = rotated_coords$y,
          fill = points_data$color[i],
          colour = points_data$border_color[i],
          size = adjusted_stroke,
          alpha = points_data$alpha[i]
        )
      }
    }
  }
  return(p)
}
