gnmFit <- function(modelTools, y, constrain, family = poisson(),
                    weights = rep.int(1, length(y)),
                    offset = rep.int(0, length(y)), nObs = length(y),
                    start, control = gnmControl(...), verbose = FALSE,
                    x = FALSE, vcov = FALSE, termPredictors = FALSE) {
    attempt <- 1
    if (verbose) width <- as.numeric(options("width"))
    repeat {
        status <- "not.converged"
        dev <- numeric(2)
        if (any(is.na(start))) {
            theta <- modelTools$start()
            theta[!is.na(start)] <- start[!is.na(start)]
            theta[constrain] <- 0
            linear <- modelTools$classID == "Linear"
            specified <- !is.na(start) | modelTools$classID == "plugInStart"
            unspecifiedLin <- linear & !specified & !constrain
            thetaOffset <- theta
            thetaOffset[!specified] <- 0
            factorList <- modelTools$factorList(thetaOffset)
            offsetSpecified <- offset + modelTools$predictor(factorList)
            X <- modelTools$localDesignFunction(thetaOffset, factorList)
            theta[unspecifiedLin] <-
                suppressWarnings(naToZero(glm.fit(X[, unspecifiedLin], y,
                                                  weights = weights,
                                                  offset = offsetSpecified,
                                                  family = family)$coef))
            oneAtATime <- !linear & !specified & !constrain
            for (iter in seq(length = control$iterStart * any(oneAtATime))) {
                if (verbose){
                    if (iter == 1)
                        cat("Running start-up iterations", "\n"[control$trace],
                            sep = "")
                    if ((iter + 25) %% width == (width - 1)) cat("\n")
                    if (!control$trace) cat(".")
                }
                for (i in rep(seq(theta)[oneAtATime], 2)) {
                    factorList <- modelTools$factorList(theta)
                    eta <- offset + modelTools$predictor(factorList)
                    mu <- family$linkinv(eta)
                    X <- modelTools$localDesignFunction(theta, factorList)
                    dmu <- family$mu.eta(eta)
                    vmu <- family$variance(mu)
                    w <- weights * dmu * dmu / vmu
                    Xi <- X[, i]
                    score <- crossprod((y - mu)/dmu, w * Xi)
                    gradient <- crossprod(w, Xi^2)
                    theta[i] <- as.vector(theta[i] + score/gradient)
                    if (!is.finite(theta[i])) {
                        status <- "bad.param"
                        break
                    } 
                }
                if (status == "not.converged") 
                    theta <- updateLinear(linear & !constrain, theta, y, offset,
                                          weights, family, modelTools, X)
                if (control$trace){
                    dev <- sum(family$dev.resids(y, mu, weights))
                    cat("Start-up iteration ", iter,
                        ". Deviance = ", dev, "\n", sep = "")
                }
                if (status == "bad.param") break
            }
            if (any(oneAtATime) & verbose & !control$trace) cat("\n")
        }    
        else theta <- structure(ifelse(!constrain, start, 0),
                                names = names(modelTools$classID))
        for (iter in seq(control$iterMax)[status == "not.converged"]) {
            if (verbose){
                if (iter == 1)
                    cat("Running main iterations", "\n"[control$trace],
                        sep = "")
                if ((iter + 21) %% width == (width - 1)) cat("\n")
                if (!control$trace) cat(".")
            }
            factorList <- modelTools$factorList(theta)
            eta <- offset + modelTools$predictor(factorList)
            X <- modelTools$localDesignFunction(theta, factorList)
            mu <- family$linkinv(eta)
            dmu <- family$mu.eta(eta)
            vmu <- family$variance(mu)
            w <- weights * dmu * dmu / vmu
            if (any(!is.finite(w))) {
                status <- "not.finite"
                break
            }
            dev[2] <- dev[1]
            dev[1] <- sum(family$dev.resids(y, mu, weights))
            if (control$trace)
                cat("Iteration ", iter, ". Deviance = ", dev[1], "\n", sep = "")
            if (is.nan(dev[1])) {
                status <- "no.deviance"
                break
            }
            z <- (y - mu)/dmu
            WX <- w * X
            score <- drop(crossprod(z, WX))
            diagInfo <- colSums(X * WX)
            if (all(abs(score) < control$tolerance*sqrt(diagInfo) |
                    diagInfo < 1e-20)){
                status <- "converged"
                break
            }
            if (iter >1 & abs(diff(dev)) < 1e-16) {
                status <- "stuck"
                break
            }
            Z <- cbind(z, X)
            WZ <- w * Z
            ZWZ <- crossprod(Z, WZ)
            ZWZinv <- MPinv(ZWZ, eliminate = 1 + modelTools$eliminate,
                            onlyFirstCol = TRUE)
            theChange <- - (ZWZinv[, 1] / ZWZinv[1, 1])[-1] 
            theta <- theta + theChange 
            theta[constrain] <- 0
        }
        if (status %in% c("converged", "not.converged") | all(!is.na(start))) {
            if (verbose) cat("\n"[!control$trace], "Done\n", sep = "")
            break
        }
        else {
            attempt <- attempt + 1
            if (verbose)
                cat(switch(status,
                           "bad.param" = "\nBad parameterisation",
                           "not.finite" = "\nIterative weights are not all finite",
                           "no.deviance" = "\nDeviance is NaN",
                           "stuck" = "\nIterations are not converging"))
            if (attempt > 5)
                stop("algorithm has failed: no model could be estimated")
            else if (verbose) cat(": restarting\n")
        }
    }
    if (status == "not.converged")
        warning("fitting algorithm has either not converged or converged\n",
                "to a non-solution of the likelihood equations: re-start \n",
                "gnm with coefficients of returned model\n")
    theta[constrain] <- NA
    if (exists("WX")) Info <- crossprod(X, WX)
    VCOV <- try(MPinv(Info, eliminate = modelTools$eliminate,
                      onlyNonElim = TRUE), silent = TRUE)
    modelAIC <- suppressWarnings(family$aic(y, rep.int(1, nObs), mu, weights,
                                            dev[1]) + 2 * attr(VCOV, "rank"))
    
    fit <- list(coefficients = theta,
                eliminate = length(modelTools$eliminate),
                predictors = eta,
                fitted.values = mu,
                deviance = dev[1],
                aic = modelAIC,
                iter = iter,
                conv = status == "converged",
                weights = w,
                residuals = z,
                df.residual = nObs - attr(VCOV, "rank"),
                rank = attr(VCOV, "rank"))
    if (x) fit$x <- structure(X, assign = modelTools$termAssign)
    if (vcov) {
        if (length(modelTools$eliminate))
            constrain <- constrain[-modelTools$eliminate]
        VCOV[constrain, constrain] <- 0
        fit$vcov <- VCOV
    }
    if (termPredictors) {
        theta[is.na(theta)] <- 0
        factorList <- modelTools$factorList(theta, term = TRUE)
        fit$termPredictors <- modelTools$predictor(factorList, term = TRUE)
    }
    fit    
}

