% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_volcano.R
\name{plot_volcano}
\alias{plot_volcano}
\title{For each provided geneset, a volcano plot of all genelist log2fc and p-values with respective geneset constituents highlighted}
\usage{
plot_volcano(
  x,
  genelist,
  plot = TRUE,
  topn_labels = 0,
  color_default = "#B0B0B0",
  color_highlight = "#ef5350",
  color_label = "#000000",
  pointsize = 2,
  pointalpha = 0.75,
  labelsize = 7
)
}
\arguments{
\item{x}{a subset of the results from \code{test_genesets} function, see example}

\item{genelist}{input genelist, must contain columns 'gene', 'log2fc' and 'pvalue_adjust' (not! log transformed). If parameter topn_labels is provided, also include a character column 'symbol' that contains gene names/symbols/labels}

\item{plot}{if \code{TRUE}, will directly show the plots. if \code{FALSE}, returns a list of ggplot objects corresponding to rows in the input \code{result} parameter}

\item{topn_labels}{for how many genes that overlap between genelist and a geneset should we plot the gene symbol? This requires a column 'symbol' in the genelist parameter (default: 0)}

\item{color_default}{color for genes that are not part of a geneset (default: grey)}

\item{color_highlight}{color used to highlight geneset constituents (default: red)}

\item{color_label}{provided that topn_labels is set, this is the color of the text labels (default: black)}

\item{pointsize}{size of the dots, this parameter is passed to geom_point (default: 2)}

\item{pointalpha}{alpha of the dots, this parameter is passed to geom_point (default: 0.75)}

\item{labelsize}{provided that topn_labels is set, this is the text size (in pt) for the labels (default: 7)}
}
\value{
if \code{plot==FALSE}, a list of ggplot2 objects. Otherwise, does not return any value
}
\description{
For each provided geneset, a volcano plot of all genelist log2fc and p-values with respective geneset constituents highlighted
}
\examples{
\donttest{
# note; this example downloads data when first run, and typically takes ~60seconds

# store the downloaded files in the following directory. Here, the temporary file
# directory is used. Alternatively, consider storing this data in a more permanent location.
# e.g. output_dir="~/data/goat" on unix systems or output_dir="C:/data/goat" on Windows
output_dir = tempdir()

## first run the default example from test_genesets() to obtain geneset results
datasets = download_goat_manuscript_data(output_dir)
genelist = datasets$`Wingo 2020:mass-spec:PMID32424284`
genesets_asis = download_genesets_goatrepo(output_dir)
genesets_filtered = filter_genesets(genesets_asis, genelist)
result = test_genesets(genesets_filtered, genelist, method = "goat",
  score_type = "effectsize", padj_method = "bonferroni", padj_cutoff = 0.05)

## example 1; select top10 GO CC terms from the geneset testing results
result_subset = result |> filter(source == "GO_CC") |> arrange(pvalue) |> head(n = 10)
pdf(paste0(output_dir, "/volcano_CC_top10.pdf"), width = 4, height = 4)
plot_volcano(result_subset, genelist)
dev.off()

## example 2;, select small genesets that are significant and have
## near-exclusive enrichment in either up up/down-regulated genes
# first, add geneset directionality scores to our results
result = score_geneset_directionality(result, genelist)
# next, subset the geneset result table
result_subset = result |>
  filter(signif & ngenes <= 50 & abs(score_directionality_rank) > 0.6) |>
  arrange(pvalue_adjust)
# finally, create plots. Note that the genelist contains a column 'symbol'
# which we use here to print labels for the topN genes per plotted geneset
pdf(paste0(output_dir, "/volcano_signif_ngenes50_directionality06.pdf"), width = 4, height = 4)
plot_volcano(result_subset, genelist, topn_labels = 10)
dev.off()
}
}
