% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv_glmnet.R
\name{cv_glmnet}
\alias{cv_glmnet}
\title{Fitting of regularized linear models}
\usage{
cv_glmnet(x, y, alpha = c("lasso", "ridge")[1], lambda = NULL,
  intercept = TRUE, thresh = 1e-10, cv_1se = TRUE, cv_nlambda = 50,
  cv_folds = NULL, cv_grouped = FALSE, cv_lambda = 10^seq(2, -3,
  length.out = cv_nlambda), cv_second = TRUE, cv_tol_second = 0.025,
  cv_log10_exp = c(-0.5, 3), cv_thresh = 1e-05, cv_parallel = FALSE,
  cv_verbose = FALSE, ...)
}
\arguments{
\item{x}{input matrix of size \code{c(n, p)}, or a vector of length
\code{n}.}

\item{y}{response matrix of size \code{c(n, q)}, or a vector of
length \code{n}.}

\item{alpha}{elastic net mixing argument in \code{\link[glmnet]{glmnet}},
with \eqn{0 \le \alpha \le 1}. Alternatively, a character string indicating
whether the \code{"ridge"} (\eqn{\alpha = 0}) or \code{"lasso"}
(\eqn{\alpha = 1}) fit is to be performed.}

\item{lambda}{scalar giving the regularization parameter \eqn{\lambda}. If
\code{NULL} (default), the optimal \eqn{\lambda} is searched by
cross-validation. If \code{lambda} is provided, then cross-validation is
skipped and the fit is performed for the given \code{lambda}.}

\item{intercept}{flag passed to the \code{intercept} argument in
\code{\link[glmnet]{glmnet}} to indicate if the intercept should be fitted
(default; does not assume that the data is centered) or set to zero
(the optimization problem above is solved as-is). Defaults to \code{TRUE}.}

\item{thresh}{convergence threshold of the coordinate descending algorithm,
passed to the \code{thresh} argument in \code{\link[glmnet]{glmnet}}.
Defaults to \code{1e-10}.}

\item{cv_1se}{shall the \emph{optimal} lambda be the \code{lambda.1se}, as
returned by \code{\link[glmnet]{cv.glmnet}}? This favors sparser fits. If
\code{FALSE}, then the optimal lambda is \code{lambda.min}, the minimizer
of the cross-validation loss. Defaults to \code{TRUE}.}

\item{cv_nlambda}{the length of the sequence of \eqn{\lambda} values,
passed to the \code{nlambda} argument in \code{\link[glmnet]{cv.glmnet}}
for the cross-validation search. Defaults to \code{50}.}

\item{cv_folds}{number of folds to perform cross-validation. If
\code{NULL} (the default), then\cr \code{cv_folds <- n} internally,
that is, leave-one-out cross-validation is performed. Passed to the
\code{nfolds} argument in \code{\link[glmnet]{cv.glmnet}}.}

\item{cv_grouped}{passed to the \code{grouped} argument in
\code{\link[glmnet]{cv.glmnet}}. Defaults to \code{FALSE}.}

\item{cv_lambda}{passed to the \code{lambda} argument in
\code{\link[glmnet]{cv.glmnet}}. Defaults to\cr
\code{10^seq(2, -3, length.out = cv_nlambda)}.}

\item{cv_second}{flag to perform a second cross-validation search if the
optimal \eqn{\lambda} was found at the extremes of the first \eqn{\lambda}
sequence (indicating that the minimum may not be reliable). Defaults to
\code{TRUE}.}

\item{cv_tol_second}{tolerance for performing a second search if
\code{second = TRUE}. If the minimum is found at the
\code{100 * cv_tol_second} lower/upper percentile of search interval, then
the search interval is expanded for a second search. Defaults to
\code{0.025}.}

\item{cv_log10_exp}{expansion of the \eqn{\lambda} sequence if the minimum
is found close to its \emph{upper} extreme. If that is the case, the
sequence for the is set as \code{10^(log10(lambda_min) + cv_log10_exp)},
where \code{lambda_min} is the minimum obtained in the first sequence. If
the minimum is found close to the lower extreme of the sequence, then
\code{-rev(cv_log10_exp)} is considered. Defaults to \code{c(-0.5, 5)}.}

\item{cv_thresh}{convergence threshold used during cross-validation in
\code{\link[glmnet]{cv.glmnet}}. Defaults to \code{1e-5}.}

\item{cv_parallel}{passed to the \code{parallel} argument in
\code{\link[glmnet]{cv.glmnet}}. Defaults to \code{FALSE}.}

\item{cv_verbose}{flag to display information about the cross-validation
search with plots and messages. More useful if \code{second = TRUE}.
Defaults to \code{FALSE}.}

\item{...}{further parameters to be passed to \code{\link[glmnet]{glmnet}}
to perform the final model fit.}
}
\value{
A list with the following entries:
\item{beta_hat}{the estimated \eqn{\mathbf{B}}{B},
a matrix of size \code{c(p, q)}.}
\item{lambda}{the optimal \eqn{\lambda} obtained by cross-validation and
according to \code{cv_1se}.}
\item{cv}{if \code{lambda = NULL}, the result of the cross-validation
search for the optimal \eqn{\lambda}. Otherwise, \code{NULL}.}
\item{fit}{the \code{glmnet} fit, computed with \code{thresh} threshold
and with an automatically chosen \eqn{\lambda} sequence.}
}
\description{
Convenience function for fitting multivariate linear models
with multivariate response by relying on \code{\link[glmnet]{cv.glmnet}}
from the \code{\link[glmnet]{glmnet-package}}. The function fits the
multivariate linear model
\deqn{\mathbf{Y} = \mathbf{X}\mathbf{B} + \mathbf{E},}{
Y = XB + E,}
where \eqn{\mathbf{X}}{X} is a \eqn{p}-dimensional vector,
\eqn{\mathbf{Y}}{Y} and \eqn{\mathbf{E}}{E} are two
\eqn{q}-dimensional vectors, and \eqn{\mathbf{B}}{B} is a
\eqn{p\times q}{p x q} matrix.

If \eqn{\mathbf{X}}{X} and \eqn{\mathbf{Y}}{Y} are \emph{centered}
(i.e., have zero-mean columns), the function estimates \eqn{\mathbf{B}}{B}
by solving, for the sample
\eqn{(\mathbf{X}_1, \mathbf{Y}_1), \ldots, (\mathbf{X}_n, \mathbf{Y}_n)}{
(X_1, Y_1), \ldots, (X_n, Y_n)}, the elastic-net optimization problem
\deqn{
\min_{\mathbf{B}\in R^{q \times p}}
\frac{1}{2n}\sum_{i=1}^n
\|\mathbf{Y}_i-\mathbf{X}_i\mathbf{B}\|^2 +
\lambda\left[(1-\alpha)\|\mathbf{B}\|_\mathrm{F}^2 / 2 +
\alpha \sum_{j=1}^p \|\mathbf{B}_j\|_2\right],
}{
\min_{B\in R^{q \times p}}
\frac{1}{2n} \sum_{i=1}^n
||Y_i - X_i B||_F^2 +
\lambda[(1 - \alpha) ||B||_F^2 / 2 +
\alpha \sum_{j=1}^p ||B_j||_2],
}
where \eqn{\|\mathbf{B}\|_\mathrm{F}}{||B||_F} stands for
the Frobenious norm of the matrix \eqn{\mathbf{B}}{B} and
\eqn{\|\mathbf{B}_j\|_2}{||B_j||_2} for the Euclidean norm
of the \eqn{j}-th \emph{row} of \eqn{\mathbf{B}}{B}. The choice
\eqn{\alpha = 0} in the elastic-net penalization corresponds to ridge
regression, whereas \eqn{\alpha = 1} yields a lasso-type estimator.
The unpenalized least-squares estimator is obtained with \eqn{\lambda = 0}.
}
\details{
If \eqn{\alpha = 1}, then the lasso-type fit shrinks to zero,
\emph{simultaneously}, all the elements of certain rows of
\eqn{\mathbf{B}}{B}, thus
helping the selection of the \eqn{p} most influential variables in
\eqn{\mathbf{X}}{X} for explaining/predicting \eqn{\mathbf{Y}}{Y}.

The function first performs a cross-validation search for the optimal
\eqn{\lambda} if \code{lambda = NULL} (using \code{cv_thresh} to control
the convergence threshold). After the optimal penalization parameter is
determined, a second fit (now with convergence threshold \code{thresh})
using the default \eqn{\lambda} sequence in \code{\link[glmnet]{glmnet}}
is performed. The final estimate is obtained via
\code{\link[glmnet]{predict.glmnet}} from the optimal \eqn{\lambda}
determined in the first step.

Due to its cross-validatory nature, \code{cv_glmnet} can be
computationally demanding. Approaches for reducing the computation time
include: considering a smaller number of folds than \code{n}, such as
\code{cv_folds = 10} (but will lead to random partitions of the
data); decrease the tolerance of the coordinate descending
algorithm by increasing \code{cv_thresh}; reducing the number of
candidate \eqn{\lambda} values with \code{nlambda}; setting
\code{second = FALSE} to avoid a second cross-validation; or considering
\code{cv_parallel = TRUE} to use a parallel backend (must be registered
before hand; see examples).

By default, the \eqn{\lambda} sequence is used with \emph{standardized}
\eqn{\mathbf{X}}{X} and \eqn{\mathbf{Y}}{Y} (both divided by their
columnwise variances); see \code{\link[glmnet]{glmnet}} and the
\code{standardized} argument. Therefore, the optimal selected \eqn{\lambda}
value assumes standardization and must be used with care if the variables
are not standardized. For example, when using the ridge analytical
solution, a prior change of scale that depends on \eqn{q} needs to be done.
See the examples for the details.
}
\examples{
## Quick example for multivariate linear model with multivariate response

# Simulate data
n <- 100
p <- 10; q <- 5
set.seed(123456)
x <- matrix(rnorm(n * p, sd = rep(1:p, each = n)), nrow = n, ncol = p)
e <- matrix(rnorm(n * q, sd = rep(q:1, each = n)), nrow = n, ncol = q)
beta <- matrix(((1:p - 1) / p)^2, nrow = p, ncol = q)
y <- x \%*\% beta + e

# Fit lasso (model with intercept, the default)
cv_glmnet(x = x, y = y, alpha = "lasso", cv_verbose = TRUE)$beta_hat
\donttest{
## Multivariate linear model with multivariate response

# Simulate data
n <- 100
p <- 10; q <- 5
set.seed(123456)
x <- matrix(rnorm(n * p, sd = rep(1:p, each = n)), nrow = n, ncol = p)
e <- matrix(rnorm(n * q, sd = rep(q:1, each = n)), nrow = n, ncol = q)
beta <- matrix(((1:p - 1) / p)^2, nrow = p, ncol = q)
y <- x \%*\% beta + e

# Fit ridge
cv0 <- cv_glmnet(x = x, y = y, alpha = "ridge", intercept = FALSE,
                 cv_verbose = TRUE)
cv0$beta_hat

# Same fit for the chosen lambda
cv_glmnet(x = x, y = y, alpha = "ridge", lambda = cv0$lambda,
          intercept = FALSE)$beta_hat

# Fit lasso (model with intercept, the default)
cv1 <- cv_glmnet(x = x, y = y, alpha = "lasso", cv_verbose = TRUE)
cv1$beta_hat

# Use cv_1se = FALSE
cv1_min <- cv_glmnet(x = x, y = y, alpha = "lasso", cv_verbose = TRUE,
                     cv_1se = FALSE)

# Compare it with ridge analytical solution. Observe the factor in lambda,
# necessary since lambda is searched for the standardized data. Note also
# that, differently to the case q = 1, no standardarization with respect to
# y happens
sd_x <- apply(x, 2, function(x) sd(x)) * sqrt((n - 1) / n)
cv_glmnet(x = x, y = y, alpha = "ridge", lambda = cv0$lambda,
          thresh = 1e-20, intercept = FALSE)$beta_hat
solve(crossprod(x) + diag(cv0$lambda * sd_x^2 * n)) \%*\% t(x) \%*\% y

# If x is standardized, the match between glmnet and usual ridge
# analytical expression does not require scaling of lambda
x_std <- scale(x, scale = sd_x, center = TRUE)
cv_glmnet(x = x_std, y = y, alpha = "ridge", lambda = cv0$lambda,
          intercept = FALSE, thresh = 1e-20)$beta_hat
solve(crossprod(x_std) + diag(rep(cv0$lambda * n, p))) \%*\% t(x_std) \%*\% y

## Simple linear model

# Simulate data
n <- 100
p <- 1; q <- 1
set.seed(123456)
x <- matrix(rnorm(n * p), nrow = n, ncol = p)
e <- matrix(rnorm(n * q), nrow = n, ncol = q)
beta <- 2
y <- x * beta + e

# Fit by ridge (model with intercept, the default)
cv0 <- cv_glmnet(x = x, y = y, alpha = "ridge", cv_verbose = TRUE)
cv0$beta_hat
cv0$intercept

# Comparison with linear model with intercept
lm(y ~ 1 + x)$coefficients

# Fit by ridge (model without intercept)
cv0 <- cv_glmnet(x = x, y = y, alpha = "ridge", cv_verbose = TRUE,
                 intercept = FALSE)
cv0$beta_hat

# Comparison with linear model without intercept
lm(y ~ 0 + x)$coefficients

# Same fit for the chosen lambda (and without intercept)
cv_glmnet(x = x, y = y, alpha = "ridge", lambda = cv0$lambda,
          intercept = FALSE)$beta_hat

# Same for lasso (model with intercept, the default)
cv1 <- cv_glmnet(x = x, y = y, alpha = "lasso")
cv1$beta_hat

## Multivariate linear model (p = 3, q = 1)

# Simulate data
n <- 50
p <- 10; q <- 1
set.seed(123456)
x <- matrix(rnorm(n * p, mean = 1, sd = rep(1:p, each = n)),
            nrow = n, ncol = p)
e <- matrix(rnorm(n * q), nrow = n, ncol = q)
beta <- ((1:p - 1) / p)^2
y <- x \%*\% beta + e

# Fit ridge (model without intercept)
cv0 <- cv_glmnet(x = x, y = y, alpha = "ridge", intercept = FALSE,
                 cv_verbose = TRUE)
cv0$beta_hat

# Same fit for the chosen lambda
cv_glmnet(x = x, y = y, alpha = "ridge", lambda = cv0$lambda,
          intercept = FALSE)$beta_hat

# Compare it with ridge analytical solution. Observe the factor in lambda,
# necessary since lambda is searched for the standardized data
sd_x <- apply(x, 2, function(x) sd(x)) * sqrt((n - 1) / n)
sd_y <- sd(y) * sqrt((n - 1) / n)
cv_glmnet(x = x, y = y, alpha = "ridge", lambda = cv0$lambda,
          intercept = FALSE, thresh = 1e-20)$beta_hat
solve(crossprod(x) + diag(cv0$lambda * sd_x^2 / sd_y * n)) \%*\% t(x) \%*\% y

# If x and y are standardized, the match between glmnet and usual ridge
# analytical expression does not require scaling of lambda
x_std <- scale(x, scale = sd_x, center = TRUE)
y_std <- scale(y, scale = sd_y, center = TRUE)
cv_glmnet(x = x_std, y = y_std, alpha = "ridge", lambda = cv0$lambda,
          intercept = FALSE, thresh = 1e-20)$beta_hat
solve(crossprod(x_std) + diag(rep(cv0$lambda * n, p))) \%*\% t(x_std) \%*\% y_std

# Fit lasso (model with intercept, the default)
cv1 <- cv_glmnet(x = x, y = y, alpha = "lasso", cv_verbose = TRUE)
cv1$beta_hat

# ## Parallelization
#
# # Parallel
# doMC::registerDoMC(cores = 2)
# microbenchmark::microbenchmark(
# cv_glmnet(x = x, y = y, nlambda = 100, cv_parallel = TRUE),
# cv_glmnet(x = x, y = y, nlambda = 100, cv_parallel = FALSE),
# times = 10)
}
}
\references{
Friedman, J., Hastie, T. and Tibshirani, R. (2010). Regularization paths for
generalized linear models via coordinate descent. \emph{Journal of
Statistical Software}, 33(1):1--22. \doi{10.18637/jss.v033.i01}
}
\author{
Eduardo García-Portugués. Initial contributions by Gonzalo
Álvarez-Pérez.
}
