% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddm.R
\name{ddm}
\alias{ddm}
\title{Double Demeaning (DDM)}
\usage{
ddm(dependent_variable, regressors, codes, robust = TRUE, data, ...)
}
\arguments{
\item{dependent_variable}{name (type: character) of the dependent variable in the dataset
\code{data} (e.g. trade flows).

This variable is logged and then used as the dependent variable in the estimation.}

\item{regressors}{name (type: character) of the regressors to include in the model.

Include the distance variable in the dataset \code{data} containing a measure of
distance between all pairs of bilateral partners and bilateral variables that should
be taken as the independent variables in the estimation.

Write this argument as \code{c(distance, contiguity, common curreny, ...)}.}

\item{codes}{variable name (type: character) of the code of the country
of origin and destination (e.g. ISO-3 codes from the variables \code{iso_o} and \code{iso_d}) in the
example datasets).

The variables are grouped by using \code{iso_o} and \code{iso_d} to obtain estimates.

Write this argument as \code{c(code origin, code destination)}.}

\item{robust}{robust (type: logical) determines whether a robust
variance-covariance matrix should be used. By default is set to \code{TRUE}.

If \code{robust = TRUE} the estimation results are consistent with the
Stata code provided at \href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook}
when choosing robust estimation.}

\item{data}{name of the dataset to be used (type: character).

To estimate gravity equations you need a square dataset including bilateral
flows defined by the argument \code{dependent_variable}, ISO codes or similar of type character
(e.g. \code{iso_o} for the country of origin and \code{iso_d} for the
destination country), a distance measure defined by the argument \code{distance}
and other potential influences (e.g. contiguity and common currency) given as a vector in
\code{regressors} are required.

All dummy variables should be of type numeric (0/1).

Make sure the ISO codes are of type "character".

If an independent variable is defined as a ratio, it should be logged.

The user should perform some data cleaning beforehand to remove observations that contain entries that
can distort estimates.

The function will remove zero flows and distances.}

\item{...}{additional arguments to be passed to \code{ddm}.}
}
\value{
The function returns the summary of the estimated gravity model as an
\code{\link[stats]{lm}}-object.
}
\description{
\code{ddm} estimates gravity models via double demeaning the
left hand side and right hand side of the gravity equation.
}
\details{
\code{ddm} is an estimation method for gravity models presented
in \insertCite{Head2014;textual}{gravity}.

Country specific effects are subdued due double demeaning. Hence, unilateral income
proxies such as GDP cannot be considered as exogenous variables.

Unilateral effect drop out due to double demeaning and therefore cannot be estimated.

\code{ddm} is designed to be consistent with the Stata code provided at
\href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook}
when choosing robust estimation.

As, to our knowledge at the moment, there is no explicit literature covering
the estimation of a gravity equation by \code{ddm} using panel data,
we do not recommend to apply this method in this case.
}
\examples{
\dontrun{
data(gravity_no_zeros)

ddm(dependent_variable = "flow", regressors = c("distw", "rta"),
codes = c("iso_o", "iso_d"),
robust = TRUE, data = gravity_no_zeros)

ddm(dependent_variable = "flow", regressors = c("distw", "rta", "comcur", "contig"),
codes = c("iso_o", "iso_d"),
robust=TRUE, data=gravity_no_zeros)
}

\dontshow{
# examples for CRAN checks:
# executable in < 5 sec together with the examples above
# not shown to users

data(gravity_no_zeros)
# choose exemplarily 10 biggest countries for check data
countries_chosen <- names(sort(table(gravity_no_zeros$iso_o), decreasing = TRUE)[1:10])
grav_small <- gravity_no_zeros[gravity_no_zeros$iso_o \%in\% countries_chosen,]
ddm(dependent_variable = "flow", regressors = c("distw", "rta"),
codes = c("iso_o", "iso_d"),
robust = TRUE, data = grav_small)
}

}
\references{
For more information on Double Demeaning as well as information on gravity
models, theoretical foundations and estimation methods in general see

\insertRef{Head2014}{gravity}

as well as

\insertRef{Anderson1979}{gravity}

\insertRef{Anderson2001}{gravity}

\insertRef{Anderson2010}{gravity}

\insertRef{Baier2009}{gravity}

\insertRef{Baier2010}{gravity}

\insertRef{Head2010}{gravity}

\insertRef{Santos2006}{gravity}

and the citations therein.

See \href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook} for gravity datasets and Stata code for estimating gravity models.
}
\seealso{
\code{\link[stats]{lm}}, \code{\link[lmtest]{coeftest}},
\code{\link[sandwich]{vcovHC}}
}
