% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sils.R
\name{sils}
\alias{sils}
\title{Structural Iterated Least Squares (SILS)}
\usage{
sils(dependent_variable, regressors, incomes, maxloop = 50,
  decimal_places = 4, robust = TRUE, verbose = FALSE, data, ...)
}
\arguments{
\item{dependent_variable}{name (type: character) of the dependent variable in the dataset
\code{data} (e.g. trade flows).

This dependent variable is divided by the
product of unilateral incomes (e.g.
GDPs or GNPs of the countries of interest, named \code{inc_o} and \code{inc_d} in the example datasets)
and logged afterwards.

The transformed variable is then used as the dependent variable in the
estimation.}

\item{regressors}{name (type: character) of the regressors to include in the model.

Include the distance variable in the dataset \code{data} containing a measure of
distance between all pairs of bilateral partners and bilateral variables that should
be taken as the independent variables in the estimation.

The distance is logged automatically when the function is executed.

Unilateral metric variables such as GDPs should be inserted via the argument \code{incomes}.

As country specific effects are subdued due to demeaning, no further unilateral variables
apart from unilateral incomes can be added.

Write this argument as \code{c(distance, contiguity, common curreny, ...)}.}

\item{incomes}{variable name (type: character) of the income of the country of
origin (e.g. \code{inc_o}) and destination (e.g. \code{inc_d}) in the dataset \code{data}.

The dependent variable \code{dependent_variable} is divided by the product of the incomes.

Write this argument as \code{c(income origin, income destination)}.}

\item{maxloop}{maximum number of outer loop iterations.
The default is set to 100. There will be a warning if the iterations
did not converge.}

\item{decimal_places}{number of decimal places that should not change after a new
iteration for the estimation to stop. The default is set to 4.}

\item{robust}{robust (type: logic) determines whether a robust
variance-covariance matrix should be used. The default is set to \code{TRUE}.

If set \code{TRUE} the estimation results are consistent with the
Stata code provided at the website
\href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook}
when choosing robust estimation.}

\item{verbose}{(type: logic) determines whether the estimated coefficients
of each iteration should be printed in the console. The default is set
to \code{FALSE}.}

\item{data}{name of the dataset to be used (type: character).

To estimate gravity equations you need a square dataset including bilateral
flows defined by the argument \code{dependent_variable}, ISO codes or similar of type character
(e.g. \code{iso_o} for the country of origin and \code{iso_d} for the
destination country), a distance measure defined by the argument \code{distance}
and other potential influences (e.g. contiguity and common currency) given as a vector in
\code{regressors} are required.

All dummy variables should be of type numeric (0/1).

Make sure the ISO codes are of type "character".

If an independent variable is defined as a ratio, it should be logged.

The user should perform some data cleaning beforehand to remove observations that contain entries that
can distort estimates.

The function will remove zero flows and distances.}

\item{...}{additional arguments to be passed to functions used by
\code{sils}.}
}
\value{
The function returns the summary of the estimated gravity model as an
\code{\link[stats]{lm}}-object. It furthermore returns the resulting coefficients for each
iteration.
}
\description{
\code{sils} estimates gravity models via
Structural Iterated Least Squares and an explicit inclusion
of the Multilateral Resistance terms.
}
\details{
\code{sils} is an estimation method for gravity models
developed by \insertCite{Head2014;textual}{gravity}.

The function \code{sils} utilizes the relationship between the Multilateral
Resistance terms and the transaction costs. The parameters are estimated by
an iterative procedure. The function executes loops until the parameters
stop changing significantly.

\code{sils} is designed to be consistent with the Stata code provided at
\href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook}
when choosing robust estimation.

As, to our knowledge at the moment, there is no explicit literature covering
the estimation of a gravity equation by \code{sils} using panel data,
we do not recommend to apply this method in this case.
}
\examples{
\dontrun{
data(gravity_no_zeros)

sils(dependent_variable = "flow", regressors = c("distw", "rta"),
incomes = c("gdp_o", "gdp_d"),
maxloop = 100, dec_places = 4, robust = TRUE, verbose = FALSE,
data = gravity_no_zeros)

sils(dependent_variable = "flow", regressors = c("distw", "rta", "comcur", "contig"),
incomes = c("gdp_o", "gdp_d"),
maxloop = 100, dec_places = 4, robust = TRUE, verbose = FALSE,
data = gravity_no_zeros)
}

\dontshow{
# examples for CRAN checks:
# executable in < 5 sec together with the examples above
# not shown to users

data(gravity_no_zeros)
# choose exemplarily 10 biggest countries for check data
countries_chosen <- names(sort(table(gravity_no_zeros$iso_o), decreasing = TRUE)[1:10])
grav_small <- gravity_no_zeros[gravity_no_zeros$iso_o \%in\% countries_chosen,]
sils(dependent_variable = "flow", regressors = c("distw", "rta"),
incomes = c("gdp_o", "gdp_d"), maxloop = 100,
dec_places = 4, robust = TRUE, verbose = TRUE, data = grav_small)
}

}
\references{
For information on \code{sils} as well as more information on gravity
models, theoretical foundations and suitable estimation methods in general see

\insertRef{Head2014}{gravity}

\insertRef{Anderson2001}{gravity}

For more information on gravity models, theoretical foundations and
estimation methods in general see

\insertRef{Anderson1979}{gravity}

\insertRef{Anderson2010}{gravity}

\insertRef{Baier2009}{gravity}

\insertRef{Baier2010}{gravity}

\insertRef{Head2010}{gravity}

\insertRef{Santos2006}{gravity}

and the citations therein.

See \href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook} for gravity datasets and Stata code for estimating gravity models.
}
\seealso{
\code{\link[stats]{lm}}, \code{\link[lmtest]{coeftest}},
\code{\link[sandwich]{vcovHC}}
}
